/*
 * File:   wctrl_main.c
 * RC Windencontroller
 * Author: Otti
 *
 * Created on 6. Januar 2018, 23:13
 * 
 * Pinbelegung und Controller-Peripherie:
 * RA0: PPM Servo Signal input
 * RA1,RA2: (AN1,AN2) static half bridge control (rot.direction), current sensing 
 * RA3: (not in use)
 * RA4,RA5: (PWMA/B) -> PWM half bridge control (power setting)
 * 
 * TMR0: Timing f. AD-Converter
 * TMR1: measure time of PPM input pulses or generate Tx/Rx bit timing for software-UART
 * TMR2: PWM motor control, milliseconds timer 
 * 
 */

// When Bootloader is used, progmas for Config-Bits are dummy (used for simulator debugging),
// because Config words cannot be changed by bootloader.
// CONFIG1
#pragma config FOSC = INTOSC    // Oscillator Selection (INTOSC oscillator: I/O function on CLKIN pin)
#pragma config WDTE = OFF       // Watchdog Timer Enable (WDT disabled)
#pragma config PWRTE = ON       // Power-up Timer Enable (PWRT enabled)
#pragma config MCLRE = OFF      // MCLR Pin Function Select (MCLR/VPP pin function is digital input)
#pragma config CP = OFF         // Flash Program Memory Code Protection (Program memory code protection is disabled)
#pragma config CPD = OFF        // Data Memory Code Protection (Data memory code protection is disabled)
#pragma config BOREN = ON       // Brown-out Reset Enable (Brown-out Reset enabled)
#pragma config CLKOUTEN = OFF   // Clock Out Enable (CLKOUT function is disabled. I/O or oscillator function on the CLKOUT pin)
#pragma config IESO = OFF       // Internal/External Switchover (Internal/External Switchover mode is disabled)
#pragma config FCMEN = OFF      // Fail-Safe Clock Monitor Enable (Fail-Safe Clock Monitor is disabled)

// CONFIG2
#pragma config WRT = OFF        // Flash Memory Self-Write Protection (Write protection off)
#pragma config PLLEN = ON       // PLL Enable (4x PLL enabled)
#pragma config STVREN = ON      // Stack Overflow/Underflow Reset Enable (Stack Overflow or Underflow will cause a Reset)
#pragma config BORV = HI        // Brown-out Reset Voltage Selection (Brown-out Reset Voltage (Vbor), high trip point selected.)
#pragma config LVP = OFF        // Low-Voltage Programming Enable (High-voltage on MCLR/VPP must be used for programming)

// #pragma config statements should precede project file includes.
// Use project enums instead of #define for ON and OFF.

#include <xc.h>
#include <stdint.h>
#include "config.h"
#include "softUART.h"
#include "ppmIn.h"
#include "motor.h"


volatile uint8_t errcode = 0;

#define DEADBAND 5

// ************************** Initialization of Clock and Peripherals
void initPeripherals(void)
{
    ANSELA = 0b00000110;		// RA1, RA2 analog (case input))
    WPUA = 0b00001001;			// RA0, RA3 = Weak Pull Up
    OSCCON = 0b01110000;		// clock = 32MHz
#ifndef __DEBUG
	while(OSCSTATbits.PLLR == 0);
#endif
    TRISA = 0b00111111;			// all IOs = input
	APFCON = 0b00000011;		// PWM (P1A, P1B) -> RA4, RA5)
	
	PWMinit();					// init PWM generator	
	softUART_init();			// setup serial communication
	UartTxByte('U');

    GIE = 1;
}





/*******************************************************************************
 * Apply deadband to control value
 *******************************************************************************/
int8_t DeadBand (int8_t ctrlval)
{
	if(ctrlval < -DEADBAND)
	{
		return ctrlval+DEADBAND;
	}
	else if (ctrlval > DEADBAND)
	{
		return ctrlval-DEADBAND;
	}
	else
		return 0;
}

/*******************************************************************************
 *						main function
 *******************************************************************************/

void main(void) {
	uint8_t timr = 0;
	initPeripherals();	
	
	SetMotor (0);
	
    while (1) 
	{
		if ((!m_ctrl)								// if motor is not spinning		
		&& (TRISAbits.TRISA1 == 0)					// and still breaking
		&& ((break_tmr - millis()) & 0x8000))		// and breaktime is over
			ReleaseBreak();

		if(uartCRbits.RXBF)
		{
			uartCRbits.RXBF = 0;
			SetMotor (uartDR);
		}
		if(processPPM ())							// new control data from input?
		{
			inCtrl = DeadBand (inCtrl);				// apply deadband
			SetMotor (inCtrl);
		}
		if(errcode)
		{
			UartTxByte(errcode);
			UartTxByte(ADRESH);						// test/diagnostics
			UartTxByte(ADRESL);
			errcode = 0;
		}
		checkCurrentTH();							// check for current exceeding threshold
		
		if((uint8_t)(timr - (uint8_t)msCnt) & 0x80)	// every 50 ms...
		{
			timr += 50;
			if (m_ctrl)								// send current value only if motor is on
			{
				UartTxByte(filt_I >> 8);
				UartTxByte(filt_I & 0xFF);
			}
		}
    }

}

