/*
 * File:   ppmIn.c
 * Decoding of PPM Servo Input Pulses
 *
 * 
 * RA0: pulse input
 * TMR1: counts microseconds
 * 
 */


#include <xc.h>
#include <stdint.h>
#include <stdbool.h>
#include "ppmIn.h"
#include "SoftUART.h"

volatile ppmCRbits_t ppmCRbits;
volatile int16_t ppmPW;
int8_t inCtrl;


/*******************************************************************************
 * setup PPM pulse processing 
 * called from software uart ISR on long input low level  
 *******************************************************************************/
void setupPPM (void)
{
	ppmCR = 0;
	// TMR1 setup:
	TMR1 = 0;
	T1CON = 0b00110001;			// PSC = 1/8 -> 1MHz, run
	PIR1bits.TMR1IF = 0;
	PIE1bits.TMR1IE = 0;		// disable TMR1 interrupt
	PPMIOCP = 1;				// IRQ on both edges
	PPMIOCN = 1;
	PPMIOCF = 0;
	INTCONbits.IOCIE = 1;		// enable PPM Pin change Interrupt (IOC)
	ppmCRbits.ENABLE = 1;
	inCtrl = 0;
}

/*******************************************************************************
 * Process PPM input pulse
 * change to software uart if there are no valid pulses
 * returns true if new control value is available
 *******************************************************************************/
bool processPPM (void)
{
	if (ppmCRbits.ENABLE)					// PPM activated?
	{
		// check if there are no PPM pulses for >65ms
		if(PIR1bits.TMR1IF)					// timer overflow every 65.5 ms
		{
			PIR1bits.TMR1IF = 0;
			if(ppmCRbits.TIMEOUT)			// >65.5 ms without a pulse?
			{
				ppmCRbits.ENABLE = 0;		// set PPM inactive
				softUART_init();			// setup UART instead			
			}
			else
				ppmCRbits.TIMEOUT = 1;
		}

		if(ppmCRbits.NEWPULSE)				// got a new PPM pulse?
		{
			ppmCRbits.NEWPULSE = 0;
			
			uint8_t temp = ppmPW >> 8;
			if(temp >= 3 && temp < 9)		// if pulsewidth is within valid range
			{
				ppmPW -= 1520-512;			// middle pw (1520) -> 512
				if(ppmPW & 0x8000)		
					ppmPW = 0;				// <0 -> 0
				ppmPW >>= 2;				// scale to inCtrl (int8)...
				
				if(ppmPW & 0x0100)			// limit to +127
					inCtrl = 127;
				else
					inCtrl = ppmPW-128;		// -> +/- 128
				
				return true;
			}
		}		
	}
	return false;
}