/*
 * File:   motor.c
 *
 * PWM generator
 * Controls motor power and direction of rotation 
 * 
 */

#include <xc.h>
#include <stdint.h>
#include "config.h"
#include "motor.h"
//#include "softUART.h"

uint24_t start_time;						// timestamp: start of motor
uint16_t break_tmr;
uint16_t CurrentTH;
int8_t m_ctrl;
volatile uint24_t msCnt;
volatile uint16_t filt_I;
volatile PWM_CTRLbits_t PWM_CTRLbits;

#ifdef SOFTPWM
volatile uint8_t tmr0h;
int16_t softpwmLow, softpwmHigh;			// pw high/low (-s))
#endif

/*******************************************************************************
 * PWM initialization
 *******************************************************************************/
void PWMinit (void)
{
	OPTION_REG = T0CKPS;		
    T2CON =   (T2CKPS << _T2CON_T2CKPS0_POSN)    // Prescaler 1:4 -> TMR2 clock = 2 MHz
			| (T2PS_VAL << _T2CON_T2OUTPS0_POSN)   // Postscaler for milliseconds counter
			| (1 << _T2CON_TMR2ON_POSN);    // TMR2 on
    PR2 = T2PERIOD-1;			// = PWM period time
#ifdef SOFTPWM
	softpwmLow = 0;
	softpwmHigh = 0;
	tmr0h = 0;
#else
	CCP1AS = 0;
  #if (SYNCPWM)
	CCP1CON = 0b10001100;		// PWM mode, P1A, P1B active-high, half bridge with dead band
//	#ifdef __DEBUG
	#if 0
		#if (T2CKPS == 0)
		PWM1CON = 12;				
		#elif T2CKPS == 1
		PWM1CON = 3;				// adjust for simulator (simulator bug!)
		#else
		PWM1CON = 0;				// adjust for simulator (simulator bug!)
		#endif
	#else
	PWM1CON = 8;				// PWM delay = 1s
	#endif
  #else
	CCP1CON = 0b00001101;		// PWM mode, P1A, P1B active-low, steering mode
	PSTR1CON = 0;				// P1A, P1B = GPIO (PWM off))
  #endif		// SYNCPWM
#endif		// SOFTPWM
	msCnt = 0;
    TMR2IE = 1;
	ADIE = 1;
    PEIE = 1;
	TMR0IE = 1;

	PWM_CTRLbits.PWM_CTRLbyte = 0;
	filt_I = 0;
	CurrentTH = I_THRESHOLD;
	
}

void ReleaseBreak (void)
{
	TRISA |= 0b00110110;				// release all H-bridge drivers
}

/*******************************************************************************
 * returns current ms counter value
 *******************************************************************************/
uint24_t millis(void) 
{
	uint24_t retval;
	do {											
		retval = msCnt;								// reading multiple bytes is not atomic!
	} while ((uint8_t)msCnt != (uint8_t)retval);	// read again if LSB has changed
    return retval;
}

/*******************************************************************************
 * Set motor power and direction
 * IN	:	power setting in percents (+/- 100%)
 *******************************************************************************/
void SetMotor (int8_t ctrl)
{
	uint16_t ctrlw;
	if(!m_ctrl)							// if motor was stopped (maybe breaking)
	{
		if(ctrl)						// new value !=0 requested?
			TRISA |= 0b00110110;		// release all H-bridge drivers
		else
		{								// ctrl = stop:
			PWM_CTRLbits.LOCK_REV = 0;	// release lock condition
			PWM_CTRLbits.LOCK_FWD = 0;
			return;						// nothing new - keep it stopped
		}
	}
		
	if(ctrl < 0)						// reverse direction:
    {		
		PWM_CTRLbits.LOCK_FWD = 0;		// unlock forward direction
		if(PWM_CTRLbits.LOCK_REV)
			return;						// reverse direction is locked
	}
	if(ctrl > 0)						// forward direction:
    {		
		PWM_CTRLbits.LOCK_REV = 0;		// unlock reverse direction
		if(PWM_CTRLbits.LOCK_FWD)
			return;						// forward direction is locked
	}
	
	m_ctrl = ctrl;
	
    if(ctrl < 0)						// <0 means reverse direction:
    {		
		PWM_CTRLbits.REVERSE = 1;
		ctrl += 100;					// P1B: (100%..0%) => (P1A: 0%..100%)
		if(ctrl < 0)
			ctrl = 0;
		if(TRISAbits.TRISA2)			// if reverse was not driven yet 
		{
			TRISAbits.TRISA1 = 1;		// release RA1 -> turns Q1 off
			__delay_us(10);				// Q1 -> off is slow...
			start_time = millis();
			ADCON1 = 0b10100000;		// right justified, Clk = fosc/32, refp=VDD
			ADCON0 = 0b00000101;		// select AD channel 1 (RA1)
		#ifdef SOFTPWM
			LATAbits.LATA4 = 0;
			LATAbits.LATA5 = 0;		
			tmr0h = 0xFE;
			PWM_CTRLbits.OUTLVL = 0;	
		#else
			CCP1ASbits.CCP1ASE = 1;		// shut down PWM for direction change
		  #if !SYNCPWM
			PSTR1CONbits.STR1A = 0;
			PSTR1CONbits.STR1B = 1;
			LATAbits.LATA5 = 0;			// PWM = P1B, P1A(RA5) = output low  
		  #endif
		#endif		// SOFTPWM
			LATAbits.LATA2 = 1;			// RA2 output high
			TRISA &= 0b11001011;		// turns Q2 on, drive PWM outputs (P1A, P1B)
		}
    }
    else								// >= 0 means forward direction:
    {									// (ctrl >= 100 -> 100% duty)
		if(ctrl > 100)
			ctrl = 100;	
		
		PWM_CTRLbits.REVERSE = 0;

		if(TRISAbits.TRISA1)			// if forward was not driven yet
		{
			TRISAbits.TRISA2 = 1;		// release RA2 -> turns Q2 off
			start_time = millis();
			ADCON1 = 0b10100000;		// right justified, Clk = fosc/32, refp=VDD
			ADCON0 = 0b00001001;		// select AD channel 2 (RA2)
	#ifdef SOFTPWM
			LATAbits.LATA4 = 0;
			LATAbits.LATA5 = 0;			
			tmr0h = 0xFE;
			PWM_CTRLbits.OUTLVL = 1;	
	#else
			CCP1ASbits.CCP1ASE = 1;		// shut down PWM for direction change
		#if !SYNCPWM
			PSTR1CONbits.STR1A = 1;
			PSTR1CONbits.STR1B = 0;
			LATAbits.LATA4 = 0;			// PWM = P1A, P1B(RA4) = output low  
		#endif
	#endif
			LATAbits.LATA1 = 1;			// RA1 output high
			TRISA &= 0b11001101;		// turns Q1 on, drive PWM outputs 
		}
    }
#ifdef SOFTPWM
	if(!m_ctrl)							// if motor is stopped: apply breaking
	{
		softpwmLow = 0;
		softpwmHigh = -10000;
		ADIE = 0;
		ADCON0 = 0;						// no AD conversions as motor is stopped
		filt_I = 0;						// measured current is now zero		
		break_tmr = millis() + BREAKDURATION;	// start timer for breaking	
		TMR0IE = 0;
		LATAbits.LATA5 = 0;				// P1A(RA5) = output low -> LS-FET off
		__delay_us(2);				
		LATAbits.LATA4 = 1;				// P1B(RA4) = output high -> HS FET on
										// -> both high side switches = on
	}
	else
	{
	// Scale pulse width with ctrl values (100% = full PWM period time)
	// on forward: active phase is softpwmLow (Q5 = on) 
	// on reverse: active phase is softpwmHigh(Q4 = on) 
		ctrlw = ((PWMPERIOD+50)/100) * (uint8_t)ctrl;
		TMR0IE = 0;
		softpwmLow = -ctrlw;
		softpwmHigh = -PWMPERIOD+ctrlw;
		if(softpwmHigh > 0)		// (for sure: >0 ?)
			softpwmHigh = 0;
	}
	TMR0IE = 1;

#else		// hardware pwm:
	if(!m_ctrl)							// if motor is stopped: apply breaking
	{
		ADIE = 0;
		ADCON0 = 0;						// no AD conversions as motor is stopped
		filt_I = 0;						// measured current is now zero		
		break_tmr = millis() + BREAKDURATION;	// start timer for breaking	

	  #if !SYNCPWM
		PSTR1CONbits.STR1A = 0;
		LATAbits.LATA5 = 0;				// P1A(RA5) = output low -> LS-FET off
		__delay_us(2);				
		LATAbits.LATA4 = 1;				// P1B(RA4) = output high -> HS FET on
										// -> both high side switches = on
	  #endif	
	}	
	ctrlw = ((T2PERIOD)*256/100) * (uint8_t)ctrl;
	CCPR1L = ctrlw >> 8;
  #ifdef Q4TIMCOMP
	#if (TMR2CLK == 8000000)			// compensation only @high PWM frequency
	if((CCPR1L > 0) && !(CCPR1L & 0x80))
		CCPR1L += Q4TIMCOMP;
	#endif
  #endif
	CCP1ASbits.CCP1ASE = 0;				// release shutdown, PWM restarts @ next cycle
	TMR0 = -10;							// force ISR to adjust ADC sample timer soon
#endif		// ifdef SOFTPWM
}

/*******************************************************************************
* check if current exceeds threshold value for shutdown
*******************************************************************************/
void checkCurrentTH(void)
{
	if(((start_time+START_INHIBIT)-millis()) & 0x800000)	// if start_time is over
	{
		if(filt_I > CurrentTH)
		{
			if(m_ctrl > 0)
				PWM_CTRLbits.LOCK_FWD = 1;	// lock for actual direction
			else
				PWM_CTRLbits.LOCK_REV = 1;
			
			SetMotor (0);					// current was above threshold: stop it!
		}
	}
}