/*
 * File:   interrupt.c
 * Interrupt service routine
 * Author: Otti
 *
 *  
 */

#include <xc.h>
#include <stdint.h>
#include "config.h"
#include "ppmIn.h"
#include "SoftUART.h"
#include "motor.h"

#define I_PROTECT 350						// AD threshold for immediate shutdown

extern volatile uint24_t msCnt;
extern volatile uint16_t filt_I;
extern volatile uint8_t errcode;

extern volatile uint8_t tmr0h;
extern int16_t softpwmLow, softpwmHigh;		// pw high/low (-s))

void __interrupt isr(void) {
	// -------------------- IOC on PPM input pin or UART Rx Startbit (same pin) ----------
    if(PPMIOCF & PPM_PIN)					
	{
		UIOCF = 0;							// reset IRQ
		if(ppmCRbits.ENABLE)				// if PPM is enabled -> PPM interrupt
		{
			if(ppmCRbits.LEVEL)				// if current level is high
			{
				if(!PPM_PORT)				// glitch filter: 
				{							// stabile low -> trailing edge of pulse 
					ppmCRbits.LEVEL = 0;
					ppmCRbits.NEWPULSE = 1;	// got a new pulse
					T1CONbits.TMR1ON = 0;	// stop timer for 16 bit reading
					ppmPW = TMR1-ppmPW;		// calculate pulse width
					T1CONbits.TMR1ON = 1;
				}
			}
			else							// if current level is low
			{
				if(PPM_PORT)				// glitch filter: 
				{							// stabile high -> leading edge of pulse 
					ppmCRbits.LEVEL = 1;
					ppmCRbits.TIMEOUT = 0;
					T1CONbits.TMR1ON = 0;	// stop timer for 16 bit reading
					ppmPW = TMR1;			// store current time
					T1CONbits.TMR1ON = 1;
				}
			}
		}
		else								// if PPM is disabled -> UART interrupt
		{									// UART Rx-Pin H->L transition - begin of startbit 
			if (!UART_PORT)					// ignore if Rx-pin is high (glitch!)
			{								
				T1CONbits.TMR1ON = 0;
				PIR1bits.TMR1IF = 0;
				UIOCN = 0;					// further Rx-interrupts from timer
				TMR1 = (-3 * BITTIME / 2)+(8*32000000/_XTAL_FREQ);	//(incl. latency compensation)
				T1CONbits.TMR1ON = 1;
				uartShR = 0x80;
				uartCRbits.RXBSY = 1;
			}
		}
	}
	//------------------------------- UART Timer interrupt -----------------------------
	else if(PIE1bits.TMR1IE && PIR1bits.TMR1IF)	// TMR1 interrupt for UART bit timing
	{
		PIR1bits.TMR1IF = 0;				// reset IRQ
		if (uartCRbits.TXBSY) 				// active transmitter
		{
			if (uartCRbits.STOPBIT)			// finished sending of stopbit
			{
				uartCRbits.STOPBIT = 0;
				T1CONbits.TMR1ON = 0;		// tx (this byte) done, stop timer
				if (uartCRbits.TXBF)		// if there are more data in buffer
				{
					UartTxNow();			// send another byte
				} 
				else						// txbuf empty -> tx complete
				{
					uartCRbits.TXBSY = 0;
					#ifdef __DEBUG
					UART_TRIS = 1;			// sim: change pin to input (->rx)
					#endif
					UIOCN = 1;				// enable rx (negative transition -> startbit)
				}
				return;
			}

			// --------- send next bit:    
			if (uartShR & 1) // send 1-bit
			{
				#ifdef __DEBUG
				UART_LAT = 1;				// simulator: push-pull -> high
				#else
				UART_TRIS = 1;				// release output (high)
				#endif	    
			} 
			else // send 0-bit:
			{
				#ifdef __DEBUG
				UART_LAT = 0;				// simulator: push-pull
				#else
				UART_LAT = 0;
				UART_TRIS = 0;				// drive output (low)
				#endif
			}

			uartShR >>= 1;
			if (uartCRbits.STARTBIT)		// if startbit was just sent:
				uartShR |= 0x80;			// set last bit = 1 = stopbit
			uartCRbits.STARTBIT = 0;
			if (!uartShR)					// if this was stopbit 
			{
				uartCRbits.STOPBIT = 1;
				#ifdef __DEBUG
				UART_TRIS = 1;				// simulator: release output
				#endif
			}

		}
		// -------------------- receive next bit:
		else if (uartCRbits.RXBSY) 
		{
			if (uartCRbits.STOPBIT) 
			{
				uartCRbits.RXBSY = 0;
				uartCRbits.STOPBIT = 0;
				UIOCN = 1;					// enable rx (negative transition -> startbit)

				if (UART_PORT)				// got stopbit: rx finished
				{
					uartDR = uartShR;
					uartCRbits.RXBF = 1;
				}
				T1CONbits.TMR1ON = 0;		// stop timer for 16 bit manipulation
				TMR1 = -BITTIME*8;			// set time before any transmission can start
				T1CONbits.TMR1ON = 1;
				return;
			}
			if (uartShR & 1)				// if this is the last data bit
				uartCRbits.STOPBIT = 1;		// expect stopbit next (= highlevel) 
			uartShR >>= 1;
			if (UART_PORT)					// read current bit value from rx port
				uartShR |= 0x80;			// received a "1"-bit
		} 
		else // extended time after Rx is over:
		{
			T1CONbits.TMR1ON = 0;			// stop timer
			if(!UART_PORT && !UIOCF)		// if input pin is still low 
				setupPPM();					// -> input looks like servo pulse
			else if (uartCRbits.TXBF)		// any data to transmit?
				UartTxNow();				// -> start transmission

			return;
		}
		// --------------- reload timer for next bit
	#if (BITTIME) >= 256		// manipulation of 16 bit timer is not atomic
		T1CONbits.TMR1ON = 0; // stop timer for 16 bit manipulation
		TMR1 -= BITTIME; // reload timer: time for next bitframe
		T1CONbits.TMR1ON = 1;
	#else
		TMR1L -= BITTIME;
		TMR1H--;
	#endif
	}
	
	// ---------------- TMR0: starts AD conversion (syncronized with PWM) ------------- 
	// in case of SOFTPWM: TMR0-ISR also handles PWM signal generation
	else if (TMR0IF && TMR0IE) 
    {
		TMR0IF = 0;

#ifdef SOFTPWM			
		if(m_ctrl)								// do only if motor is powered
		{
			if((tmr0h == 0xFF) || ADIE)			// reached end of pulse or AD interrupt was activated
			{
				if (!PWM_CTRLbits.REVERSE		// forward direction
				&& PORTAbits.RA5)				// and P1A output high (outlevel = low, Q5 = on)
					ADCON0bits.ADGO = 1;		// start AD conversion

				if (PWM_CTRLbits.REVERSE		// reverse direction
				&& PORTAbits.RA4)				// and P1B output high (outlevel = high, Q4 = on)
					ADCON0bits.ADGO = 1;		// start AD conversion
			}
			ADIE = 1;
		}
		
		if(++tmr0h == 0)						// timeout of 16-bit timer (tmr0h:TMR0)
		{										// (time to toggle output level)
			int16_t temp16 = -10000;			// default if no switching: next action after 10ms

			if(PWM_CTRLbits.OUTLVL == 1)		// if current output level is high
			{
				if (softpwmLow != 0)			// and low pulse exists
				{
					temp16 = softpwmLow;		// load PW (low)
					PWM_CTRLbits.OUTLVL = 0;	// change output to low
					LATAbits.LATA5 = 0;			// H+L FET off for switching
					LATAbits.LATA4 = 0;
				}
			}
			else								// if current output level is low
			{
				if (softpwmHigh != 0)			// and high pulse exists
				{
					temp16 = softpwmHigh;		// load PW (high)
					PWM_CTRLbits.OUTLVL = 1;	// change output to high
					LATAbits.LATA5 = 0;			// H+L FET off for switching
					LATAbits.LATA4 = 0;
				}
			}		
			temp16 += TMR0;						// load timer (tmr0h:TMR0) with new PW
			TMR0 = temp16 & 0xFF;
			tmr0h = temp16>>8;
			if((tmr0h & 0x80) == 0)				// new timer value >= 0:
			{
				tmr0h = 0xFF;					// timeout immediately!
				TMR0IF = 1;
			}
			
			if(PWM_CTRLbits.REVERSE)
			{
				if(PWM_CTRLbits.OUTLVL)			// high == active phase
					LATAbits.LATA4 = 1;			// turn HS-FET (Q4) on
			#if SYNCPWM	
				else							// low == freewheeling phase
					LATAbits.LATA5 = 1;			// synchronous mode: turn the other FET on
			#endif
			}
			else								// (dir == forward)
			{
				if(PWM_CTRLbits.OUTLVL == 0)	// low == active phase
					LATAbits.LATA5 = 1;			// turn LS-FET (Q5) on
			#if SYNCPWM	
				else							// high == freewheeling phase
					LATAbits.LATA4 = 1;			// synchronous mode: turn the other FET on
			#endif
			}
		}
	}
		
#else	// ---------------  hardware PWM:
		// trigger AD conversion while PWM is "on",
		// otherwise we would get an invalid reading!
		// sets TMR0 to trigger next AD sample at the end of next PWM "on" phase
		if(m_ctrl)									// do only if motor is powered
		{
			uint8_t porta = PORTA;					// read current PWM output level
			ADIF = 0;
			ADIE = 0;
			ADCON0bits.ADGO = 1;					// start AD conversion
			porta &= PORTA;							// current PWM output level should still be high
			uint8_t temp = ((uint8_t)(T2PERIOD-1)-TMR2);	// = TMR2 ticks till start of next PWM cycle 
		#if (T0CKPS)
			if(--temp == 0xFF)						// adjustment: avoids IRQ beeing too late due to
				temp += T2PERIOD;					// reset of TMR0's prescaler on write op. 
		#endif
			temp >>= 1;
													
			if(PWM_CTRLbits.REVERSE)
			{
				if(porta & 1<<4)					// if P1B output (RA4) was high:
					ADIE = 1;						// enable ADC interrupt
													// else discard result of conversion				
				if(temp < T2PERIOD/4)				// if TMR2 was read before PR2 reached:
					temp += T2PERIOD/2;				// add a full period time
			}
			else									// rot.direction = forward
			{
				if(porta & 1<<5)					// if P1A output (RA5) was high:
					ADIE = 1;						// enable ADC interrupt
													// else discard result of conversion			
				temp += CCPR1L>>1;					// -> time till end of P1A high pulse
				if (temp > 3*T2PERIOD/4)			// adjust to about 1 PWM period from now
					temp -= T2PERIOD/2;
			}

		#if ((TMR2CLK/TMR0CLK) == 4)
			temp >>= 1;
		#endif
			temp += (T2PERIOD * (49/PWMPERIOD))/(TMR2CLK/TMR0CLK);	// <50s: skip periods - ADC sample every 2nd|3rd period only
			TMR0 -= temp;			
		}
	}
#endif										// (SOFTPWM)

    // ----------------------------------------------------- AD conversion done:
    else if (ADIF) 
	{
		uint16_t adresult = ADRES;
		ADIF = 0;
		if (adresult > I_PROTECT) 
		{
			LATA = 0;
			TRISA |= 0b00000110;			// RA1,RA2 = input
			errcode = 0xE7;
			m_ctrl = 0;
			
	#ifdef SOFTPWM
			softpwmLow = 0;					// disable software PWM
			softpwmHigh = 0;
	#else
			CCP1ASbits.CCP1ASE = 1;			// shut down PWM 
		#if !SYNCPWM
			PSTR1CONbits.STR1A = 0;			// steering: both PWM pins = port, low
			PSTR1CONbits.STR1B = 0;
		#endif
	#endif
			PWM_CTRLbits.SHORTDET = 1;		// set error flag
		}
		filt_I -= filt_I / 8;
		filt_I += adresult;
    }
	// ----------------------------------------------------- Milliseconds Counter
    // TMR2 with postscaler: IRQ every millisecond or every half ms (PWM > 16kHz)
    else if (TMR2IF) 
	{
	#if (PWMFREQ > 16000)
		static uint8_t T2postcounter = 1;
	#endif

		TMR2IF = 0;		// clr IRQ
		
	#ifdef SOFTPWM
		msCnt++;							// increment milliseconds
	#elif (PWMFREQ > 32000)					// >32kHz: TMR2-ISR rate = PWMFREQ/16
		T2postcounter -= 16;
		if(T2postcounter & 0x80)			// <0
		{
			T2postcounter += PWMFREQ/1000;
			msCnt++;						// increment milliseconds
		}
	#elif (PWMFREQ > 32000)					// >32kHz..64kHz: TMR2-ISR rate = 4kHz
		if(--T2postcounter == 0)			
		{
			T2postcounter = 0x04;			// every 4th ISR-call:
			msCnt++;						// increment milliseconds
		}
	#elif (PWMFREQ > 16000)					// >16kHz..32kHz: TMR2-ISR rate = 2kHz
		T2postcounter--;
		if(T2postcounter & 0x01)			// every 2nd ISR-call:
			msCnt++;						// increment milliseconds
	#elif (PWMFREQ >= 1000)					// 1kHz .. 16kHz: TMR2-ISR rate = 1kHz
		msCnt++;							// increment milliseconds
	#elif (PWMFREQ == 500)
		msCnt += 2;
	#endif					
    }
}

