/*
 * File:   config.h
 * 
 * Defines for General Configuration Settings and Options
 */

#ifndef _CONFIG_H_
#define _CONFIG_H_

#define PWMFREQ 32000      // PWM output frequency (Hz)
                            // valid options for using hardware PWM (kHz): 
                            //      1, 2, 4, 5, 8, 10, 16, 20, 32, 40, 50, 64, 80, 100, 125 
                            // software PWM: any random frequency in range of 32Hz...1kHz
                           

#define SYNCPWM 0         // turns on opposite MOSFET in PWM off phase
                            // options: 1=yes (only recommended for high inductance (mH) loads)
                            // options: 0= no (low inductance loads in H range)
#define SYNCDELAY 8         // switching delay HS<->LS FET, s/8 (only for SYNCPWM == 1)

#define Q4TIMCOMP 4         // compensates for slower turn-off time of Q4 for approx. same pulse width
                            // of Q4 vs. Q5. (s/8)

#define BREAKDURATION 1000  // on stop: breaking applied for this time (ms)

#define START_INHIBIT 200   // time(ms): inhibits overcurrent-shutdown on motor startup 

#define I_THRESHOLD 0x380   // current threshold for motor shutdown 

#define _XTAL_FREQ 32000000	// clock oscillator frequency (Fosc)
                            // options: 32000000

#if (PWMFREQ >= 1000)                           // use hardware PWM:
    #if (PWMFREQ > 32000)
        #define T2PS_VAL (15)                   // T2 postscaler value (IRQ every 16th PWM period)
    #elif (PWMFREQ > 16000)
        #define T2PS_VAL (PWMFREQ/2000-1)		// T2 postscaler value (IRQ every 0.5ms)
    #elif (PWMFREQ >= 1000)
        #define T2PS_VAL (PWMFREQ/1000-1)		// T2 postscaler value (IRQ every 1ms)
    #elif (PWMFREQ == 500)
        #define T2PS_VAL (0)                    // T2 postscaler value = 1 (IRQ every 2ms)
    #endif

    #if (PWMFREQ >= 32000)                      // (range: >= 32kHz) 
        #define TMR2CLK (8000000)
        #define T2CKPS (0)                      // TMR2 prescaler: /1
        #define TMR0CLK (4000000)               // TMR0 clk = 2MHz 
        #define T0CKPS (0)                      // TMR0 PS: /2
    #elif (PWMFREQ >= 8000)                     // (range: 8kHz..31kHz) 
        #define TMR2CLK (2000000)
        #define T2CKPS (1)                      // prescaler: /4
        #define TMR0CLK (1000000)               // TMR0 clk = 1MHz 
        #define T0CKPS (2)                      // TMR0 PS: /8
    #elif (PWMFREQ >= 2000)                     // (range: 2kHz..7kHz) 
        #define TMR2CLK (500000)
        #define T2CKPS (2)                      // prescaler: /16
        #define TMR0CLK (250000)                // TMR0 clk = 0.25 MHz 
        #define T0CKPS (4)                      // TMR0 PS: /32
    #elif (PWMFREQ == 1000)                     // (1kHz) 
        #define TMR2CLK (125000)
        #define T2CKPS (3)                      // prescaler: /64
        #define TMR0CLK (62500)                 // TMR0 clk = 62.5 kHz 
        #define T0CKPS (6)                      // TMR0 PS: /128

    #endif

    #define T2PERIOD (TMR2CLK/PWMFREQ)          // PWM periode length (TMR2 ticks) 

#else                                           // PWM frequency is low, use Software PWM:
    #define SOFTPWM
    #define TMR2CLK (2000000)
    #define T2CKPS (1)                          // prescaler: /4
    #define TMR0CLK (1000000)                   // TMR0 clk = 1MHz
    #define T0CKPS (0b010)                      // (prescaler: /8)
    #define T2PERIOD (250)                      // TMR2 period = 2MHz/250 = 8kHz
    #define T2PS_VAL (7)                        // T2 postscaler value: /8 -> 1kHz for ms counter
#endif

    #define PWMPERIOD (1000000/PWMFREQ)         // PWM periode length (s))

#endif  // _CONFIG_H_