/*
 * File:   SoftUART.c
 * Software-UART fr Debug-Zwecke
 * Author: Otti
 *
 * 
 * RA0: serial Data I/O (half duplex, open drain)
 * TMR1: Baudrate-Generator
 * 
 */


#include <xc.h>
#include <stdint.h>
#include "config.h"
#include "ppmIn.h"
#include "SoftUART.h"

volatile uint8_t uartDR, uartShR;
volatile uartCRbits_t uartCRbits;
uint8_t txidx_wr, txidx_rd;
uint8_t txbuf[UTXBUFSIZE];

/*******************************************************************************
 * Initialization/Activation of software UART
 * 
 *******************************************************************************/
void softUART_init(void) {
	uartCR = 0;
	txidx_wr = 0;
	txidx_rd = 0;
	UIOCN = 1;				// enable rx negative transition IRQ
	UIOCF = 0;

	// TMR1 setup:
	PIE1bits.TMR1IE = 1;
	T1CON = 0b00110000;		// PSC = 1/8 -> 1MHz, keep stopped for now
	PIR1bits.TMR1IF = 0;
	INTCONbits.PEIE = 1;	// enable TMR1 interrupt
	INTCONbits.IOCIE = 1;	// enable UART Rx-Interrupt (IOC)
#ifdef __DEBUG
	UART_LAT = 1;			// simulator: Push/Pull
	UART_TRIS = 0;
#endif
}

/*******************************************************************************
 * Put byte into tx buffer for transmission
 * data will be sent immediately or after rx transaction is complete
 *******************************************************************************/
void UartTxByte(uint8_t data) 
{
	if(!ppmCRbits.ENABLE)			// if pin is in PPM-mode -> discard
	{
		PIE1bits.TMR1IE = 0;		// no Tx interrupt while modifying txidx
		txbuf[txidx_wr] = data;
		txidx_wr++;					// put byte into tx buffer, tx later
		txidx_wr &= UTXBUFSIZE - 1;
		uartCRbits.TXBF = 1;		// "(more) data to send from tx buffer"
		PIE1bits.TMR1IE = 1;
		if (T1CONbits.TMR1ON == 0)	// if Rx or Tx process is not yet active
		{
			PIR1bits.TMR1IF = 1;	// T1ISR will start transmission
		} 
	}
}

/*******************************************************************************
 * start transmission of data from tx buffer
 * 
 *******************************************************************************/
void UartTxNow(void) 
{
	uartShR = txbuf[txidx_rd];
	txidx_rd++;
	txidx_rd &= UTXBUFSIZE - 1;
	if (txidx_rd == txidx_wr)
		uartCRbits.TXBF = 0;			// tx buffer is now empty
	uartCRbits.TXBSY = 1;				// transmitting
	uartCRbits.STARTBIT = 1;			// begin startbit...
	UIOCN = 0;							// disable Rx (IOC-interrupt))
	UIOCF = 0;

	TMR1 = -(BITTIME-3*32000000/_XTAL_FREQ);	// (incl. latency compensation)
	T1CONbits.TMR1ON = 1;
	UART_LAT = 0;					// output low for startbit
	UART_TRIS = 0;
}
