//*****************************************************************************
//*	
//*
//*		relocator.cpp
//*
//*
//*****************************************************************************

#include <stdio.h>
#include <stdlib.h>
#include "target.h"

#define COD_GOTO 0x2800			// goto instruction code (add address, 11 bits)
#define COD_CALL 0x2000			// call instruction code (add address, 11 bits)
#define COD_MOVLP 0x3180		// movlp instruction code (add data, 7 bits)


#define PCLATH_MASK 0x78		// bits of PCLATH that are used for call,goto
#define JMPADR_MASK 0x7FF		// operand mask (call and goto)
		

typedef enum {
	STDINST,		// instr. does nothing special regarding relocation process
	GOTO,
	CALL,
	BRA,
	BRW,
	MOVLW,
	MOVW_PCLATH,
	MOVLP,	
	BSF_PCLATH,
	BCF_PCLATH
} PIC_IT;

//==============================================================================
// returns the type-ID of instruction and stores argument value from instruction.
PIC_IT instrType(FLASH opcode, int *arg)
{
	*arg = 0;
	switch (opcode)
	{
	case 0x018A:						// clrf PCLATH
		return MOVLP;				// treat like movlp 0
		
	case 0x000B:
		return BRW;
		
	case 0x0189:					// clrf WREG
		return MOVLW;				// treat like "movlw 0"
		
	case 0x008A:					// movwf PCLATH
		return MOVW_PCLATH;
	default:
		break;
	}
	
	*arg = opcode & 0x07FF;			// 11 bits operand
	switch (opcode & 0x3800) 		// test if goto or call
	{
	case 0x2000: 
		return CALL;
	case 0x2800:
		return GOTO;
	default:
		break;	
	}

	*arg = opcode & 0x1FF;			// 9 bits operand
	if((opcode & 0x3E00) == 0x3200)	// bra instruction?
		return BRA;
	
	*arg = opcode & 0xFF;			// 8 bits operand
	if((opcode & 0x3F00) == 0x3000)
		return MOVLW;

	*arg = opcode & 0x7F;			// 7 bits operand
	if((opcode & 0x3F80) == COD_MOVLP)
		return MOVLP;
			 	
	
	*arg = (opcode & 0x0380) >> 7;	// arg = bit number
	switch (opcode & 0x3C7F) 		// test if bsf or bcf bit,PCLATH
	{
	case 0x100A: 
		return BCF_PCLATH;
	case 0x140A:
		return BSF_PCLATH;
	default:
		break;	
	}
	
	return STDINST;
}

//==============================================================================
// returns != 0 if picflash[location] contains an instruction
// which may skip the following instruction

int isSkipInstr (int location)
{
	if (location >= 0)
	{
		if((picflash[location] & 0x3800) == 0x1800)
			return 1;		// = btfss or btfsc
		if((picflash[location] & 0x3B00) == 0x0B00)
			return 1;		// = decfsz or incfsz
	}
	return 0;
}

//==============================================================================
// writes goto or call instruction into picflash[*pcadr++]

void putJump(int jmpdest, int *pcadr, int opcode, SFR *pclath)
{
	if(((jmpdest >> 8) ^ *pclath) & PCLATH_MASK)		// if it is a jump to another memory bank: 
	{
		
		*pclath = (jmpdest >> 8) & PCLATH_MASK;
		if(isSkipInstr (*pcadr-1))		// cannot skip two instructions:
		{											// put movlp first, then skipping instruction
			picflash[*pcadr] = picflash[*pcadr-1];
			picflash[*pcadr-1] = COD_MOVLP | *pclath;
		}
		else
			picflash[*pcadr] = COD_MOVLP | *pclath;		// must write PCLATH before call|goto
		(*pcadr)++;
		
	}
	picflash[*pcadr] =	(opcode & ~JMPADR_MASK) | (jmpdest & JMPADR_MASK);
	(*pcadr)++;
}

//==============================================================================
// relocates min. first 2 words of application's startup code (loc.0x000)
// to memory location (patchadr).
// assumptions:
// - there is no "brw" instr. to relocate,
// - application does not call/jump directly to intructions modified by relocation, 
// - some weird code might not get properly relocated, e.g. a loop right at reset vector
//   or conditional skipping modification of PCLATH 
 
int codeRelocator (int patchadr, int rvsize)
{
	int src = 0;
	int dest = patchadr;
//	int i=0, 
	int state=0;
	int arg, temp;
	PIC_IT itype;
	
	SFR pclath_src = 0;			// tracks PCLATH SFR register content (orig.)
	SFR pclath_dest = (patchadr >> 8) & PCLATH_MASK;	// tracks PCLATH content of patch code 
	SFR wreg;
		
	do
	{
		itype = instrType(picflash[src], &arg);	
		if(isSkipInstr (src))			// if instruction may skip one:
			state = 1;							// treat both instructions as one unit				
			
		switch(itype)
		{
		case MOVLW:
			wreg = arg;
		case STDINST:
			picflash[dest++] = picflash[src++];
			break;		// any instruction without effect on relocation: just copy it!
						
		case BRA:
			arg += 1+src;		// calculate dest.address
			if((pclath_src != 0) && (instrType(picflash[arg],&temp) != MOVLP))
			{
				printf("\nRelocator: GOTO 0x%03X results in PCLATH = 0, from original code PCLATH = 0x%02X\n",arg, pclath_src);
				exit(EXIT_FAILURE); 
			}
			putJump(arg, &dest, COD_GOTO, &pclath_dest);
			if(!isSkipInstr (src-1))					// if not a conditional goto:
				return dest;							// relocation complete
			src++;
			break;				// relocate instruction after skip

		case GOTO:
			arg |= (pclath_src & PCLATH_MASK) << 8;		// final destination address
			putJump(arg, &dest, COD_GOTO, &pclath_dest);
			if(!isSkipInstr (src-1))					// if not a conditional goto:
				return dest;							// relocation complete
			src++;
			break;				// relocate instruction after skip
			
		case CALL:
			arg |= (pclath_src & PCLATH_MASK) << 8;		// final destination address
			putJump(arg, &dest, COD_CALL, &pclath_dest);
			src++;
			break;
	
		case BCF_PCLATH:
			pclath_src &= ~(1<<arg);
			src++;
			break;
			
		case BSF_PCLATH:
			arg = pclath_src | 1<<arg;

		case MOVLP:
			src++;
			pclath_src = arg;
			break;
			
		case MOVW_PCLATH:
			src++;
			pclath_src = wreg;
			break;
			
		default:
		case BRW:			// relocation of brw not implemented
			printf("\nRelocator: cannot relocate BRW instruction!\n",arg);
			exit(EXIT_FAILURE); 
		}
		
		if(state > 0)		// > 0: cannot interrupt instruction sequence here!
			state--;		
		else
		{
			if((pclath_src != 0) && (instrType(picflash[src], &arg) != MOVLP))	// any PCLATH contents given by original code?
			{
				if(src <= rvsize)
					continue;	// cannot put preceeding MOVLP there, relocate one more instruction 
				src--;
				picflash[src] = COD_MOVLP + pclath_src;
			}
			if(src >= rvsize)		// reserve bootloder resetvector code
			{	
				putJump(src, &dest, COD_GOTO, &pclath_dest);	// put there a jump to original code
				return dest;							// relocation complete
			}
		}
	} while (dest < (patchadr+8));

	return dest;
}


