/*******************************************************************************
  PC Command Line Tool for use with PIC bootloader

  Version 0.1			(C) 2015 Thomas Elger (thomas@picalic.de)
  
  This program is free software; you can redistribute it and/or modify it 
  under the terms of the GNU General Public License as published by the 
  Free Software Foundation; either version 3 of the License, or (at your 
  option) any later version.
  This program is distributed in the hope that it will be useful, but 
  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY 
  or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License 
  for more details.

  For a copy of GNU General Public License, see <http://www.gnu.org/licenses/>. 
/*******************************************************************************/



#include <stdio.h>
#include <stdlib.h>
#include <string.h>
//#include <conio.h>
#include "Options.h"
#include "Comport.h"
#include "hexread.h"
#include "BLcom.h"
#include "target.h"
#include "relocator.h"

#define MAJ_VERSION 0			// program release version
#define MIN_VERSION 9

extern FILE *logfile;

void exitfunction(void)
{
	if(resetflag != 0)
		send2bl(CMDRESET);		// reset target
	ComClose();
	if(logflag)
	{
		fprintf(logfile,"\n--------------- end of log --------------\n");
		fclose(logfile);
	}
}

// *****************************************************************************
// for debugging: print row of data words
void printwrdata(char *rwdata)
{
	int i;
	for(i=0; i<picdata.rowsize; i++)
	{
		if((i & 0x07) == 0)
			printf("\n-> %04x: ", i + (((rwdata[2] & 0xFF) << 8) | (rwdata[1] & 0xFF)));
		printf(" %04x", ((rwdata[2*i+4] & 0xFF) << 8) | (rwdata[2*i+3] & 0xFF)); 
	}
}

//********************************************************************
// check for valid memory range
int validMem (int address)
{
	if((picdata.ee_size != 0)
	&& (address >= picdata.eeaddr)
	&& (address < picdata.eeaddr+picdata.ee_size))	// within eeprom range
		return 1;	// memory contains a data byte
	if((address < 0x8000) && (address >= picdata.flashsize))		// above program flash, but not config
		return 0;	// no memory there
	
	// if patchcode preceeds bootloader code: do not consider patchcode = "read only"!
	if((address >= ((picdata.bl_start+picdata.rowsize-1)/picdata.rowsize*picdata.rowsize))
		 && ((address < picdata.bl_end) || (picdata.bl_end == 0)))	// would overwrite bootloader
		return -1;	// memory is read only
	if(address >= (0x8000+picdata.rowsize))	// above config
		return 0;
	return 2;		// memory contains a data word
}

//********************************************************************
// read memory row (size= flash erase block) from target
void readMemRow (int address)
{
	int i;
	if(validMem(address))
	{
		address &= ~(picdata.rowsize-1);		// = first address of row
		for(i=0; i<picdata.rowsize; i++)
		{
			picflash[address+i]= readMemWord (address+i);
		}
	}
}

//********************************************************************
// calculate and store application's checksum data
void calcAppChksum(int maxaddr)	
{
	int i, chksum, picapp_end;
	
	// determine end of program code section for checksum calculation:
	picapp_end = picdata.xchk;				// default: set end address by parameter
	if(picdata.xchk == 0)					
		picapp_end = maxaddr;					// automatic setting from hex file data
	if(picdata.xchk == 1)									
		picapp_end = picflash[picdata.app_start+3]-1;	// end address given in code section		
	if(picapp_end >= picdata.flashsize)		
		picapp_end = picdata.flashsize-1;		// limit to flash size
	if((picdata.bl_start > 0) && (picapp_end >= picdata.bl_start))
		picapp_end = picdata.bl_start-1;		// limit to below start of bootloader code

	if(picdata.bl_start==0)
	{ 	
		if((picdata.bl_version & 0x7F) != 0)		// bl version >0 requires app.length rather than endaddress
			picflash[picdata.app_start+3] = 1+picapp_end-picdata.app_start;
		else
			picflash[picdata.app_start+3] = picapp_end+1;
	}
	
	for(i=picdata.app_start; i<picapp_end; i++)		// for memory area within checksum section: 
		picflash[i] &= 0x3FFF;						// force erasing memory that is not modified by hexfile 

	if(picdata.bl_start == 0)
		picflash[picdata.app_start+2] = 0;			// checksum will be stored here, initialize to 0 for calculation

	chksum = 0;
	for(i=picdata.app_start;i<=(picapp_end);i++)	// calculate application's checksum
		chksum = (0x4000 + chksum - picflash[i]) & 0x3FFF;
		

	if(picdata.bl_start)
	{
		if((picdata.bl_version & 0x7F) != 0)		// chksum data located @patchcode+7 on version >0 
		{
			picflash[picdata.patchcode+6] = picapp_end+1;	// app_length == app_end, since app starts @0
			picflash[picdata.patchcode+7] = chksum;
		}
		else
		{
			picflash[picdata.patchcode+6] = chksum;
			picflash[picdata.patchcode+7] = picapp_end+1;
		}
	}
	else
	{
		picflash[picdata.app_start+2] = chksum;
	}
	
	if(infoflag)
		printf("\nApplication checksum range: 0x%04X-0x%04X\n", picdata.app_start, picapp_end);
}


//*****************************************************************************
// calculate and store installer's checksum data
//
// Installer code is used when loading a new bootloader to change
// the reset vector, which is write-protected by the bootloader, to the 
// new bootloader's start address.
// How it works:
// -->	Hex file contains both, new bootloader and installer code.
// --> 	active bootloader loads hex data into flash and starts new bootloader
// 		as application. 
// -->	new bootloader detects the installer code as valid application (if 
//		checksum is correct) and executes it
// -->	installer modifies reset vector to jump into new bootloader

void calcInstallerChksum(void)					
{
	char signature[] = "!LdrIns!";
	int i, addr;
	int chksum=0;
	// search for signature:
	for(addr=0; addr < picdata.flashsize; addr++)
	{
		i = 0;
		while (picflash[addr+i] == signature[i])
			i++;
		if(i > 7)
			break;			// found signature at picflash[addr]
	}
	if(addr >= picdata.flashsize)
		return;				// Installer not found
		
	if(picflash[addr+10] == picdata.hw_id)
	{
		printf("\nInstalling new bootloader...\n");
	
		if(picdata.bl_start == 0)
		{
			for(i=0; i<picdata.bl_end;i++)
				picflash[i] = readMemWord(i);
		}
		
		i = picflash[addr+9];			// length of application code
		addr = picflash[addr+8];		// set startaddress
		do{
			chksum = (0x4000 + chksum - picflash[addr]) & 0x3FFF;
			addr++;
		} while (--i > 0);
		picflash[addr] = chksum;
		picflash[addr+1] = -chksum & 0x3FFF;		// application checksum should stay the same
	}
	else
	{
		printf("\nHardware-ID mismatch, check PIC device type setting!");
		printf("\nInstallation of new bootloader rejected\n");
		exit(EXIT_FAILURE);
	}
}

// *****************************************************************************
// 					m a i n 
// *****************************************************************************
int main(int argc, char** argv) 
{
	FILE *inputfile = NULL;
	char *inputfilename;
	int i,j, temp, limit;
	char ch;
	char rwdata[2*MAXROWSIZE+5];
	
	atexit(exitfunction);

	for(i=0;i<MAXFLASHSIZE;i++) picflash[i] = 0xFFFF;		// initialize flash data to "empty"

	picdata.flashsize = 0x800;		// init to defaults...
	picdata.rowsize = 16;
	picdata.bl_start = 0;
	picdata.xchk = 0;
	picdata.app_start = 0x100;
	picdata.debugflag = 0;
	picdata.ee_size = 0;
	strcpy(picdata.picname,"(unknown)");


	inputfilename = processOptions(argc, argv);		// read config from file and set options from command line 

	if(infoflag)
		printf("%s Version %d.%d\n",progname,MAJ_VERSION,MIN_VERSION);
	
//	char test[3] = {0,1,2};
//	while(1)
//	{
//		send2bl(test,3);
//	}
		
	if(picdata.debugflag)
	{
		picflash[0] = 0x318E;	// movlp 0x0E
		picflash[1] = 0x2EE0;	// goto 0x06E0 (-> 0x0EE0)
//		picflash[0] = 0x1E03;
//		picflash[1] = 0x2900;
		
		picflash[2] = 0x3188;
		picflash[3] = 0x3188;
		picflash[4] = 0x3188;
		picflash[5] = 0x3188;

//		picflash[0x0700] = 0x1E03;
//		picflash[0x0701] = 0x2FF8;		
		picflash[0x0EE0] = 0x1E03;
		picflash[0x0EE1] = 0x2FF8;
		
//		picflash[0x8006] = 0x1BC0;			// PIC12LF1840
//		picflash[0x8006] = 0x1B84;			
	}

	if(GetTargetInfo() < 0)
	{
		printf("\ntarget not connected!\n");
		exit(EXIT_FAILURE);
	}
	else
		if(CheckCompatibility())
			exit(EXIT_FAILURE);

	//-------------------------------------------------------------
	// option RD: Read data from target
	if(read_from >= 0)
	{
		if (read_to < 0)
			read_to = read_from;
		if (read_to < read_from)
			read_to = MAXFLASHSIZE-1;
		if (read_to >= MAXFLASHSIZE)
			read_to = MAXFLASHSIZE-1;
			
		for(i=0;read_from<=read_to; read_from++)
		{
			temp = validMem(read_from);
			if(temp != 0) 						// skip non-existing memory areas
			{
				if((read_from & (temp==1?0x0F:0x07)) == 0)	// force newline at 8 or 16 words boundary
					i=0;									// (depending on section's word size)
				if(i==0)
					printf("\n  %04x: ", read_from);		// start a new line with address
				if(temp == 1)								// range is data flash: size = 8 bits
					printf(" %02x", readMemWord (read_from) & 0xFF); 
				else 										// else words are 14 bits 
					printf(" %04x", readMemWord (read_from)); 
				i++;
			}
		}
	}
	//-------------------------------------------------------------
	// option ERASE: fill specified memory range with 0x3FFF 
	if(erase_from >= 0)
	{	
		if ((erase_from/picdata.rowsize*picdata.rowsize) != erase_from)	// row boundary?
			readMemRow(erase_from);								// no, save contents row for partial erase
		if ((erase_to | picdata.rowsize-1) != erase_to)			// row boundary?
			readMemRow(erase_to);								// no, save contents row for partial erase
					
		for(i=erase_from;i<=erase_to;i++) 
		{
			if(validMem(i)>0)
				picflash[i] = 0x3FFF;		// MSB = 0: "has to be erased"
//				picflash[i] = 0x5555;		// MSB = 0: "has to be erased"
		}
	}
	//-------------------------------------------------------------
	// option FILL: fill specified memory range with given number
	if(fill_from >= 0)
	{	
		if ((fill_from/picdata.rowsize*picdata.rowsize) != fill_from)	// row boundary?
			readMemRow(fill_from);								// no, save contents of row for partial erase
		if ((fill_to | picdata.rowsize-1) != fill_to)			// row boundary?
			readMemRow(fill_to);								// no, save contents of row for partial erase
					
		for(i=fill_from;i<=fill_to;i++) 
		{
			if(validMem(i)>0)
				picflash[i] = fill_data;
		}
	}

	if (infoflag)
	{
		printf("\nTarget device ID:      0x%04X = %s", picdata.hardware_id, &picdata.picname);
		printf("\nProgram flash size:    0x%04X", picdata.flashsize);
		if(picdata.ee_size)
			printf("\nData EEPROM:           0x%04X - 0x%04X", picdata.eeaddr, picdata.eeaddr-1+picdata.ee_size);
		printf("\nFlash erase block:     0x%04X", picdata.rowsize);
		printf("\nBootloader ID:         0x%02X", picdata.bl_version);
		printf("\nBootloader location:   0x%04X", picdata.bl_start);
		if(picdata.bl_end != 0)
			printf(" - 0x%04X", picdata.bl_end-1);
		printf("\nTarget's user ID:      %04X", picdata.user_id);
		if(picdata.bl_start == 0)
		{
			printf("\nApplication program memory space: 0x%04X - 0x%04X", picdata.bl_end, picdata.flashsize-1);
			if(picdata.bl_version & 0x80)
			{
				printf("\nProgram memory words 0x%04X, 0x%04X", picdata.bl_end+2, picdata.bl_end+3);
				printf(" are used for checksum data,\ntherefore application code must not use these memory locations!");
			}
		}
		else
			printf("\nApplication program memory space: 0x%04X - 0x%04X", 0, picdata.bl_start-1);		
		if (picdata.bl_version & 0x80)
		{
			if(picdata.xchk == 0)
				printf("\nXCHK = 0: Application checksum range given by max. used flash.");
			if(picdata.xchk == 1)	
				printf("\nXCHK = 1: App. checksum range given by word (start+3) in app.data.");
			if(picdata.xchk > 1)
				printf("\nXCHK = 0x%04X: App. checksum range 0x%04X to 0x%04X.", picdata.xchk, (picdata.bl_start>0)?0:picdata.bl_end, picdata.xchk);
	
		}


	}

	//-------------------------------------------------------------
	// option hex file: read hex file 
	if(inputfilename != NULL)
	{
		inputfile = fopen(inputfilename, "r"); 
		if(inputfile == NULL)
		{
			printf("\ncannot open hex input file!\n");
			exit(EXIT_FAILURE);
		}
		else
		{
			temp = readHexFile(inputfile,picflash,picdata.flashsize);
			fclose(inputfile);			
			if (temp < 0)
			{
//				printf("\nerror in hex data!\n");
				exit(EXIT_FAILURE);
			}
			else
			{	
				if((picflash[picdata.app_start] & picflash[picdata.app_start+1]) != 0xFFFF)	
				// do only if hexfile contains startup code, else it seems to be a data section  
				{
					if(picdata.bl_start)						// bootloader in upper memory:
					{
						limit = picdata.patchcode+8;			// 8 words reserved for patchcode
						if(picdata.bl_version & 0x80)			// if bootloader uses validation by checksum:
							limit -= 2;							// last two words reserved for checksum
						if(codeRelocator (picdata.patchcode, picdata.rvsize) > limit)		// relocate app's reset entry code 
						{
							printf("\nRelocation of applications reset entry code failed!\n");
							return 1;
						}
						picflash[0] = readMemWord (0);		// (reset vector) copy goto or movlp from device's flash
						if(picdata.bl_start+8 > 0x7FF)		// if BL is located beyond -> movlp+goto: 2 instructions
							picflash[1] = readMemWord (1);
					}
										
					if(picdata.bl_version & 0x80)			// if bootloader uses validation by checksum:
						calcAppChksum(temp);				// calculate the application's checksum
	
					calcInstallerChksum();					// If the bootloader installs another bootloader version:
															// checksum must be valid to execute installer code! 
				}
				resetflag = 1;
			}	
		}
	}
		
	//---------------------- program the PIC
		
	if((inputfilename != NULL) || erase_from >= 0 || fill_from >= 0 )			// if hex file was read or erase/fill address given: write flash! 
	{
		printf("\nProgramming target flash...");
		for(i=picdata.app_start & ~(picdata.rowsize-1); i<MAXFLASHSIZE; i+=picdata.rowsize)			// loop row by row...
		{
			// create data record for current row -> rwdata:
			rwdata[0] = CMDWRITE;
			rwdata[1] = i & 0xFF;			// addr. low
			rwdata[2] = i >> 8;				// addr. high
			temp = 0xFFFF;
			for(j=0; j<picdata.rowsize; j++)
			{
				rwdata[2*j+3] = picflash[i+j] & 0xFF;		// data low
				rwdata[2*j+4] = (picflash[i+j] >> 8) & 0x3F; // data high
				temp &= picflash[i+j];						// MSB = 0 if any word of row contains data
			}
			if((temp & 0x8000) == 0)								// do not program if whole row is empty
			{
				if(picdata.debugflag)
					printwrdata(rwdata);
				
				rwdata[2*picdata.rowsize+3] = 0;					// initialize checksum of record
				for(j=1; j<(2*picdata.rowsize+3); j++)				// calculate checksum
					rwdata[2*picdata.rowsize+3] -= rwdata[j];
					
				if((j = send2bl(rwdata,2*picdata.rowsize+4)) != 0) 
				{
					printf("Tx Error! %x\n", j);
					exit(EXIT_FAILURE);		// abort if tx failed
				}
				if(rxByteLog() != ACK)	
				{
					printf("\nNAK or Timeout Error!\n");
					exit(EXIT_FAILURE);		// expect ACK 
				}
			}
		}
		printf("Ok!\n");
//		resetflag = 1;
	}
	return 0;
}
