//*****************************************************************************
//*
//*		Options.cpp
//*		
//*		Process options from command line and configuration file
//*
//*****************************************************************************
//

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include "Comport.h"
#include "target.h"
#include "Options.h"


FILE *logfile;
FILE *configfile;
int connected=0, logflag=0, infoflag=0, resetflag=0;
int portnum=1, baudrate=115200; 
int erase_from=-1, erase_to=-1;
int fill_from=-1, fill_to=-1, fill_data;
int read_from=-1, read_to=-1;
char progname[30];		// just program name, without extension
char progpath[256];		// path with name, without extension

//#include <iostream>

// *****************************************************************************
// display help text, then exit(0)

void printhelp(void)
{
	printf("\n**********************************************************************\n");
	printf("**   Uploads hexfile to PIC microcontroller with bootloader         **\n");
	printf("**********************************************************************\n");
	printf("\nUsage: %s [option [option]...] [<hexfile>]\n", progname);
	printf("Example: %s /COM4 /BAUD=9600 blinky.hex\n", progname);
	printf("\nOptions:\n");
	printf("/BAUD=<baudrate>   *  set baud rate for serial communication\n");
	printf("/COM<n>            *  select COM port for serial communication\n");
	printf("/EEPROM=<from-to>  *  specify range of data EEPROM\n"); 
	printf("/ERASE<from-to>       erase PIC's memory in given address range\n");
	printf("/FILL<from-to,value>  fill memory (from-to) with (value)\n");
	printf("/FLASHSIZE=<words> *  set size of PIC's program memory\n");
	printf("/INFO                 show information about target\n");
	printf("/LOG                  log serial communication process into %s.log\n",progname);
	printf("/READ<from-to>        read out and display data from PIC's memory\n");
	printf("/ROWSIZE=<nwords>  *  set flash erase block size \n");
	printf("/RUN                  run application by performing target reset\n");
	printf("/XCHK=<0|1|addr>   *  exclude memory > addr from checksum calculation\n");
	printf("                      (0: addr=auto, 1: addr=(RESET+3) in hex file)\n");
	printf("\nSettings marked with (*) are stored in %s.cfg and used from\n",progname);
	printf("this file, until settings are modified by command line options.\n");
	printf("If possible, %s will obtain FLASHSIZE and ROWSIZE automatically\n", progname);
	printf("from PIC's hardware ID, overriding settings from options or cfg-file.\n");
	printf("Numbers may be given in decimal (nn) or hexadecimal (0xnn) notation.\n");  
	
	exit(EXIT_SUCCESS);
}


//***************************************************************************** 
// converts decimal or hexadecimal number from ASCII string to integer

int dechextoi(char *str)
{
	int retval = 0, i=0;
	while(str[i]=='0')				// skip leading zeros
		i++;
	if(isdigit(str[i]))
		return atoi(&str[i]);		// it's a decimal number!
	if(toupper(str[i]) == 'X')		// 'X' denotes hexadecimal radix
		i++;
	while(isxdigit(str[i]))
	{
		retval <<= 4;
		if (isdigit(str[i]))
			retval |= str[i]-'0';
		else
			retval |= toupper(str[i]) + 10 -'A';
		i++;
	}
	return retval;
}

//***************************************************************************** 
// converts string (<from>-<to>,<value>) to address range and value
// returns value or -1 if there is no "...,value"
int RangeVal(char *str, int *from, int *to)
{
	char *tempp;
	int retval = -1;
	*from = dechextoi(str);
	if((tempp = strchr(str,'-')) != NULL)		// if "-" is there: 
	{
		if(isdigit(*(tempp+1)))						// if "to" value is given
			*to = dechextoi(tempp+1);
		else
			*to = MAXFLASHSIZE-1;					// "from-" means <from> to top of flash
	}
	else											// "to" is not specified:
		*to = *from;								// address range is single address
	
	if (*to >= MAXFLASHSIZE)
		*to = MAXFLASHSIZE-1;
		
	if((tempp = strchr(str,',')) != NULL)		// if "," is there:
		retval = dechextoi(tempp+1);
	return retval;
	
}

// *****************************************************************************
// evaluate parameter from string
// return values: 0 = ok, 1 = overriding option from config file

int EvalParameter(char *str)
{
	int temp;

	if(strncasecmp("COM",str,3)==0)
	{
		portnum = atoi(&str[3]);
		return 1;
	}
	if(strncasecmp("BAUD=",str,5)==0)
	{
		baudrate = atoi(&str[5]);
		return 1;
	}
	if(strncasecmp("EEPROM=",str,7)==0)
	{
		RangeVal(&str[7], &picdata.eeaddr, &temp);
		picdata.ee_size = (temp+1) - picdata.eeaddr;
		return 1;
	}

	if(strncasecmp("ERASE",str,5)==0)
	{
		RangeVal(&str[5], &erase_from, &erase_to); 
		return 0;
	}
	if(strncasecmp("FILL",str,4)==0)
	{
		fill_data = RangeVal(&str[4], &fill_from, &fill_to); 
		return 0;
	}
	if(strncasecmp("FLASHSIZE=",str,10)==0)
	{
		picdata.flashsize = dechextoi(&str[10]);
		return 1;
	}
	if(strncasecmp("ROWSIZE=",str,8)==0)
	{
		picdata.rowsize = dechextoi(&str[8]);
		return 1;
	}
	if(strncasecmp("READ",str,4)==0)
	{
		RangeVal(&str[4], &read_from, &read_to); 
		return 0;
	}
	if(strncasecmp("LOG",str,3)==0)
	{
		logflag = 1;
		return 0;
	}
	if(strncasecmp("INFO",str,4)==0)
	{
		infoflag = 1;
		return 0;
	}
	if(strncasecmp("RUN",str,3)==0)
	{
		resetflag = 1;
		return 0;
	}
	if(strncasecmp("XCHK=",str,5)==0)
	{
		picdata.xchk = dechextoi(&str[5]);
		return 1;
	}
	if(strcasecmp("DEBUG",str) == 0)
	{
		picdata.debugflag = 1;
		return 0;
	}
	if(strcasecmp("HELP",str) == 0)
	{
		printhelp();
		return 0;
	}
	printf("unknown option: %s\n",str);
	exit(EXIT_FAILURE);

}

// *****************************************************************************
// process config file and command line options (main parameters) 
// returns pointer to name of file from command line (=arg without slash) or NULL.

char* processOptions(int argc, char** argv)
{	
	char *filearg = NULL;
	FILE *fileptr;
	int cfgflag = 0;
	char string[256];
	int i;
	
	// copy program name without extension and without path (if any)
	// into progname, progpath = path+progname
	strcpy(progpath,argv[0]);
	for(i=strlen(progpath); i > 0; i--)
	{
		if(progpath[i] == '.')
			progpath[i] = 0;	// strip extension
		if(progpath[i-1] == '\\')
			break;				// i -> start of filename
	}
	strcpy(progname, &progpath[i]);

	if(argc<2)
	{
		printhelp();			// no further options: print help
	}							// (printhelp will exit from the program)
	
	//-------------------------------------------------------------
	// read settings from config file first:
	
	strcpy(string,progpath);
	strcat(string,".cfg");
	fileptr = fopen(string, "r"); 
	if(fileptr != NULL)
	{
		while(fgets(string, 50,fileptr))
			EvalParameter(string);
		fclose(fileptr);	
	} 
	
	//-------------------------------------------------------------
	// process command line arguments:
	for(i=1;i<argc;i++)
	{
		if(argv[i][0] != '/')
			filearg = argv[i];				// no slash, assume this arg specifies hex file
		else 
		cfgflag += EvalParameter(&argv[i][1]);
	}

	//-------------------------------------------------------------
	// if any parameter has changed: save data in config file:
	if(cfgflag)
	{
		strcpy(string,progpath);
		strcat(string,".cfg");
		configfile = fopen(string,"w");
		fprintf(configfile,"COM%d\n",portnum);
		fprintf(configfile,"BAUD=%d\n",baudrate);
		fprintf(configfile,"EEPROM=0x%x-0x%x\n",picdata.eeaddr,picdata.eeaddr-1+picdata.ee_size);
		fprintf(configfile,"FLASHSIZE=0x%x\n",picdata.flashsize);
		fprintf(configfile,"ROWSIZE=0x%x\n",picdata.rowsize);
		fprintf(configfile,"XCHK=0x%x\n",picdata.xchk);
		fclose(configfile);
	}

	//-------------------------------------------------------------
	// open com port:
	if(!ComOpen(portnum,baudrate))
	{
		printf("\nCan not open port COM%d!\n",portnum);
		exit(EXIT_FAILURE);
	}

	//-------------------------------------------------------------
	// open log file if logging is enabled:
	if(logflag)
	{
		strcpy(string,progpath);
		strcat(string,".log");
		logfile = fopen(string,"w");
		if(logfile == NULL)
			logflag = 0;			// could not open logfile, reset logflag
	}
	return filearg;
}
