// *****************************************************************************
// *
// *    Routines for communication with bootloader
// *
// *****************************************************************************
//#define WIN32_LEAN_AND_MEAN
#include <conio.h>
#include <stdio.h>
#include <windows.h>
#include <string.h>
#include <time.h>
#include "Comport.h"
#include "target.h"
#include "BLcom.h"

extern int logflag;
extern FILE *logfile;
//int writereturn = 0;
int fullduplexflag = 0; 

// *****************************************************************************
// write byte transfer into logfile
void logwrite(int data, int dir)
{
	static int current_dir = 0;
	static int count = 0;
		
	if (logflag == 0)
		return;				// logging is off
		
	if((dir != current_dir) || (++count > 16))	// data direction change or time for a newline?
	{
		current_dir = dir;
		count = 0;
		if(dir != 0)		// != 0 means "Tx"
			fprintf(logfile,"\nTx:");
		else
			fprintf(logfile,"\nRx:");		
	}
	fprintf(logfile," %02x", (unsigned char) data);
}


// *****************************************************************************
// Receive byte from comport.
// Returns received data or -1 on timeout. 
int rxByte(int timeout)
{
	int d, timecnt = 0;
	while(ComGetReadCount() == 0)
//	while((d = ComRead()) < 0)
	{
		if(++timecnt > timeout)			// timeout?.
		{
			if(logflag)	fprintf(logfile," !!! Timeout !!!");
			return -1;
		}
		Sleep(5);
	}
	return ComRead();
}

// *****************************************************************************
// Receive byte from comport.
// Returns received data or -1 on timeout. 
int rxByteLog(void)
{
	if(picdata.debugflag)
		return ACK;		// simulator: return ACK

	int d = rxByte(100);
	if(d >= 0)
		logwrite(d,0);
	return d;
}


// *****************************************************************************
// sends data from buffer or single byte to bootloader.
// Reads back, expecting the same data (half duplex communication!).
// returns 0 if read data matches transmitted data.
int send2bl(char data)
{
	if(picdata.debugflag)
		return 0;
	while(ComGetReadCount())
		ComRead();				// flush Rx-buffer
	ComWrite(data);
	logwrite(data,1);
	if(fullduplexflag != 0)
		return 0;
	else
		return data-rxByte(10);	
}
  
int send2bl(char *buffer, int count)
{
	int i, j, retval;
	unsigned char c;
		
	if(picdata.debugflag)
		return 0;
		

//	Sleep(20);
	for(j=0; j<3; j++)			// loop: retries if data could not be properly sent
	{
		while(ComGetReadCount())
			ComRead();				// flush Rx-buffer
		i=ComWrite(buffer, count);	// send new data
		retval = 0;

		if(fullduplexflag == 0)
		{
			for(i=0;i<count;i++)		// read echoed data (half duplex!)
			{
				c = rxByte(5);
				retval += (unsigned char)buffer[i] - c;
				logwrite(buffer[i],1);
				if ((unsigned char)buffer[i] != c)
					logwrite(c,0);
			}
		}
		if(retval == 0)
			break;				// transmission succeeded!
		else
			putchar('!');		// fail, retry...
	}
	return retval;
}

// *****************************************************************************
// reads word from connected PIC's memory

int readMemWord (int address)
{
	char txdata[3];
	int rxdata;
	int temp;

	if(picdata.debugflag)
	{
		if(address >= 2)
			return picflash[address];
		else
			return resetvect[address];		// simulator: reset vector from simulated MPU
	}
	
	txdata[0] = CMDREAD;		// issue Read command to bootloader
	txdata[1] = address & 0xFF;
	txdata[2] = address >> 8;
	if(send2bl(txdata, 3) != 0) return -1;
	rxdata = rxByteLog();
	if(rxdata < 0)
		exit(EXIT_FAILURE);
	temp = rxByteLog();
	if(temp < 0)
		exit(EXIT_FAILURE);
	rxdata |= temp << 8;
	return rxdata; 
}

// *****************************************************************************
// Test it communication line is single wire/half duplex or two wire/full duplex:
void FullDuplexCheck (void)
{
	while(ComGetReadCount())
		ComRead();				// flush Rx-buffer
	ComWrite(0x5A);				// send some testpattern
	Sleep(50);
	if(ComRead() != 0x5A)
		fullduplexflag = 1;		// if char was not echoed back: assume full duplex connection
	else
		fullduplexflag = 0;		// if char was echoed back: assume half duplex connection	
}

// *****************************************************************************
// connect with bootloader 
// returns bootloader version on success, else -1
// note: calling FullDuplexCheck() multiple times is necessary, because a running 
// PIC application may pull the Tx/Rx-line low before reset, this leads to false 
// detection of full duplex connection! 

int connectBL (void)
{
	int i, ch;
	int timestamp;
	
	if(picdata.debugflag)
		return 0x80+BL_SUPPORT;
	
	// Test it communication line is single wire/half duplex or two wire/full duplex:
	FullDuplexCheck ();
	
	send2bl(CMDVERSION);		// check if bootloader is already online...
	ch = rxByteLog();
	if((ch == ACK) && fullduplexflag)
	{
		rxByteLog();				// (ignore, but log)
		FullDuplexCheck();			// check again!
		send2bl(CMDVERSION);		// check if bootloader is already online...		
		ch = rxByteLog();
	}
	if (ch == ACK)				// if bootloader is already online:
		return rxByteLog();		// done!
	
	for(i=0; i<70; i++)			// for sure:
	{
		ComWrite(0);			// send a bunch of zeros to flush any pending data packet
		ComRead();
	}
	Sleep(50);
	FullDuplexCheck();			// flush buffer and check again
	
	send2bl(CMDVERSION);		// check if bootloader is already online...
	ch = rxByteLog();
	if((ch == ACK) && fullduplexflag)
	{
		rxByteLog();				// (ignore, but log)
		FullDuplexCheck();			// check again!
		send2bl(CMDVERSION);		// check if bootloader is already online...		
		ch = rxByteLog();
	}
	if (ch == ACK)				// if bootloader is already online:
		return rxByteLog();		// done!
		
	printf("connecting with bootloader...\n(please connect|reset target or press any key to abort)\n");
	timestamp = clock();
	while(1)
	{
		if(kbhit() || ((clock() - timestamp)/CLOCKS_PER_SEC > 40))
			return -1;					// abort by user or timeout: not connected!

		if((ch = rxByteLog()) == ACK)	// bootloader "hello"
			break;
	}
	FullDuplexCheck();			// flush buffer and check again
	send2bl(CMDVERSION);		// check if bootloader is online...

	if(rxByteLog () == ACK) 	
		return (rxByteLog());		// received ACK: connected -> return bl-version		
	else
		return -1;
}

// *****************************************************************************
// call repeatedly in order to keep connection with bootloader alive
void keepAlive (void)
{
	if(picdata.debugflag)
		Sleep(100);
	else
	{
		ComWrite(0);
		Sleep(100);
		rxByte(5);
	}
}


