;**************************************************************************
;***			Bootloader for Enhanced Midrange PIC		***
;**************************************************************************
;
; Version 0.8		(C) 2015,2016 Thomas Elger (thomas@picalic.de)
; 
; This program is free software; you can redistribute it and/or modify it 
; under the terms of the GNU General Public License as published by the 
; Free Software Foundation; either version 3 of the License, or (at your 
; option) any later version.
; This program is distributed in the hope that it will be useful, but 
; WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY 
; or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License 
; for more details.
;
; For a copy of GNU General Public License, see <http://www.gnu.org/licenses/>. 
;**************************************************************************

;if bootloader is located in first memory section (address 0),
;application program memory contents (APP_START = 0x0100) is:
;0x100: Reset entry application code
;0x101: Reset entry application code
;0x102: checksum data (sum of all words of application memory & 0x3FF = 0) 
;0x103: Length: length of application section for checksum calculation 
;0x104: interrupt service routine
;...

;if bootloader is located in upper memory section, application is compiled/assembled as if
;there were no bootloader. When uploading the application code, the "loader"-tool patches the 
;applications reset entry into 8 words below bootloader start address. 
;
;When Application code starts, some registers are modified by the bootloader and will
;therefore not contain their initial values as specified in the datasheet.

    radix dec		    ;default radix: decimal 
; __________________________________________________________________________
;|									    |
;|		     Defines for Build Configuration:			    |
;|__________________________________________________________________________|
	
;define LED_PORT and LED_PIN (NLED_PIN for active low) in order to enable 
;use of a LED, indicating bootloader activity: 
;#define LED_PORT	PORTC	
;#define NLED_PIN	3		

;define BOOT_PORT and BOOT_PIN|NBOOT_PIN in order to enable
;jumping into bootloader if pin is pulled high|low on reset: 
;#define BOOT_PORT	PORTA	
;#define NBOOT_PIN	1		
 
;------ define COM_PORT/COM_PIN for single wire, half duplex communication line:
#define COM_PORT	PORTA		;comment out if full duplex connection is used!	
#define COM_PIN	5
;------ for full duplex (2-wire, Rx/Tx): define separate Tx/Rx Ports & Pins:
;(comment out these four lines if single wire/half duplex connection is used)	
;#define RX_PORT	PORTA		
;#define RX_PIN		0
;#define TX_PORT	PORTA
;#define TX_PIN		1
;------	

;#define EUSART				;use EUSART rather than software UART	
#define LOADERBAUD	115200		;COM baudrate 
#define ENABLE_CHECKSUM			;execute app-code only if it is checksum-validated
;#define BL_LOCATION 0			;uncomment to place BL at start of Flash
;#define BL_LOCATION 0x700
;(if no BL_LOCATION is defined here, bootloder will be located at top of flash)

;#define RESVECTOR	0x100		;installer startup address 
;Define RESVECTOR only if bootloader has to be built for upper memory and shall be
;installed by bootloader that actually runs in lower memory location.  
;_____________________________________________________________________________
;=>   PIC16F1705
#ifdef __16F1705	
  #include <p16F1705.inc>         	; processor specific variable definitions
  #define ROWSIZE 32			; size of flash row (words)
  #define FLASHSIZE 0x2000		; total program flash (words) 
  #define CODEROMSIZE 0x1F80		; size of flash (words)to use for program code
  					; (in this case: reserve 0x1F80...0x1FFF (HEF) for data storage)
  #define HARDWARE_ID 0x3055		; (for verification of matching PIC type with installer code)
  __config _CONFIG1, _FOSC_INTOSC & _WDTE_SWDTEN & _PWRTE_ON & _MCLRE_ON & _CP_OFF & _BOREN_ON & _IESO_OFF & 0x3FFF
  __config _CONFIG2, _WRT_OFF & _PLLEN_ON & _BORV_LO & _LVP_OFF & 0x3FFF
#endif
;__________________________________________________________________________________________
;=>   PIC12F1840
#ifdef __12F1840	
  #include <p12F1840.inc>         	; processor specific variable definitions
  #define ROWSIZE 32			; size of flash row (words)
  #define FLASHSIZE 0x1000		; total program flash (words) 
  #define CODEROMSIZE 0x1000		; size of flash (words)to use for program code
  #define EE_ORG 0xF000			; EEPROM startaddress 
  #define HARDWARE_ID 0x1B80		; (for verification of matching PIC type with installer code)
  __config _CONFIG1, _FOSC_INTOSC & _WDTE_OFF & _PWRTE_ON & _MCLRE_ON & _CP_OFF & _BOREN_ON & _IESO_OFF & 0x3FFF
  __config _CONFIG2, _WRT_OFF & _PLLEN_ON & _BORV_HI & _LVP_OFF & 0x3FFF
#endif
;__________________________________________________________________________________________
;=>   PIC16F1503
#ifdef __16F1503
  #include <p16F1503.inc>         	; processor specific variable definitions 
  #define ROWSIZE 16			; size of flash row (words)
  #define FLASHSIZE 0x800		; total program flash (words) 
  #define CODEROMSIZE 0x7E0		; size of flash (words)to use for program code
					; (e.g. in this case: reserve 0x7E0...0x7FF for data storage)
  #define HARDWARE_ID 0x2CE0		; (for verification of matching PIC type with installer code)

  __config _CONFIG1, _FOSC_INTOSC & _WDTE_SWDTEN & _PWRTE_ON & _MCLRE_ON & _CP_OFF & _BOREN_NSLEEP & 0x3FFF
  __config _CONFIG2, _WRT_OFF & _LVP_OFF & _BORV_LO & 0x3FFF
#endif
;__________________________________________________________________________________________
;=>   PIC16F1825
#ifdef __16F1825	
  #include <p16F1825.inc>         	; processor specific variable definitions
  #define ROWSIZE 32			; size of flash row (words)
  #define FLASHSIZE 0x2000		; total program flash (words) 
  #define CODEROMSIZE 0x2000		; size of flash (words)to use for program code
  #define EE_ORG 0xF000			; EEPROM startaddress 
  #define HARDWARE_ID 0x2760		; (for verification of matching PIC type with installer code)
  __config _CONFIG1, _FOSC_INTOSC & _WDTE_OFF & _PWRTE_ON & _MCLRE_OFF & _CP_OFF & _BOREN_ON & _IESO_OFF & 0x3FFF
  __config _CONFIG2, _WRT_OFF & _PLLEN_ON & _BORV_HI & _LVP_OFF & 0x3FFF
#endif
;__________________________________________________________________________________________

;#define SIMULATOR			;(for debugging with simulator)
  
 ifndef BL_LOCATION			;bl in higher memory aerea:
;Calculate size of bootloader code in order to set proper start address
BL_SIZE set 0xC4
  if CODEROMSIZE == FLASHSIZE
     #define BL_ON_TOP			;no gap between bootloader and top of memory
  endif
  ifdef BL_ON_TOP
BL_SIZE -= 8
  endif
  ifdef EUSART
BL_SIZE -= 14
  endif
  ifdef EECON1
BL_SIZE += 5
  endif
  ifdef ENABLE_CHECKSUM
BL_SIZE += 0x19			;with checksum calc., bl is longer
  endif
BL_SIZE set (BL_SIZE+ROWSIZE)/ROWSIZE*ROWSIZE	;(align)
BL_LOCATION set (CODEROMSIZE-BL_SIZE)	;start of bootloader code
 endif

 if BL_LOCATION == 0
  #define APP_START 0x100		;start of application program
 else
  #define APP_START 0			;BL resides in higher memory range
  #define PATCH_ROW (BL_LOCATION+(BL_SIZE-ROWSIZE))	;patching application reset code @end of BL
 endif

#define ACK 0x06
#define NAK 0x15
#define BL_VERSION 1	;bits 0..6: version info,
			;bit 7: checksum calculation (app.code) enabled

 ifndef PMADRL				;MPU types with EEPROM use alternate register names:
#define PMADRL EEADRL
#define PMADRH EEADRH
#define PMDATL EEDATL 
#define PMDATH EEDATH 
#define PMCON1 EECON1
#define PMCON2 EECON2
 endif

 ifdef COM_PORT
#define TX_PORT COM_PORT		;case half duplex:
#define TX_PIN COM_PIN			;TX_PIN and RX_PIN are the same!
#define RX_PORT COM_PORT
#define RX_PIN COM_PIN
 endif

;__________________________________________________________________________________________
 
 errorlevel -302,-306			;avoid lots of warnings about banking

FOSC set 16000000			;clock oscillator frequency
 
; ------ Calculation for software UART baudrate generator:
BITTIME set (FOSC/4)/LOADERBAUD-2	;(for software UART)
T0_DIV set 1				;TMR0 division factor
T0_PSV set 0x0F				;TMR0 prescaler setting (start: no prescaler)
 while (BITTIME*3/2) > 255		;1.5*BITTIME must not exceed 255!
T0_PSV set (T0_PSV+1)&0x07
T0_DIV <<= 1
BITTIME set (FOSC/(4*T0_DIV))/LOADERBAUD-1 
 endw

 ; ------ Calculation for EUSART baudrate generator: 
BAUDGEN set (FOSC/2)/LOADERBAUD 
 if (BAUDGEN & 1)
BAUDGEN set BAUDGEN/2
 else
BAUDGEN set BAUDGEN/2-1
 endif

;RAM:

	CBLOCK 	0x70		;common RAM: 
;Alignment & Order of appLength,chksum,dummy is important!	
appLength:	2		;length of app.section (count down)
chksum:	    	2
dummy:		2		;FlashRead will put data here while checking appcode 

temp				;temporary storage
flags				;some status flags
timeoutCnt: 2			;up-counter for Rx-Timeout
s_data				;register for serial data I/O
halfDplxCnt			;>1 if Rx = echo from single wire expected
highaddr			;address bits 8..15 (as given in hex file)
								
	ENDC

;definitions of flags Bits:
#define VALIDAPP 0		;assume application program is ok
;#define TIMEOUT 1		;timeout occured while waiting for Rx data
#define RDFLG 2			;set for read-cmd, clr for write


LEDOFF 	macro
 #ifdef LED_PORT
   #ifdef NLED_PIN
	bsf	LED_PORT,NLED_PIN
   #else
	bcf	LED_PORT,LED_PIN
   #endif
 #endif
	endm

LEDON 	macro
 #ifdef LED_PORT
   #ifdef NLED_PIN
	bcf	LED_PORT,NLED_PIN
   #else
	bsf	LED_PORT,LED_PIN
   #endif
 #endif
	endm

;__________________ Macro: moves single bit with predictable timing ____________
;(always 5 cycles, last instruction modifies destination bit)
 
movbit 	macro 	src, srcbit, dest, destbit	

	movf	dest,W			;current status of dest.-bit -> W
	btfsc	src,srcbit		;if source == 0: keep dest.level W, XOR will give 0
	xorlw	1<<destbit		;source == 1: toggle bit, so XOR will give 1
	andlw	1<<destbit		;(do not modify the other bits)
	xorwf	dest,F			;modify destination bit (or not ;) )
		
	endm

LOGROWSIZE set 1
 while (1<<LOGROWSIZE < ROWSIZE)	;get log2(ROWSIZE) for bitnumber
LOGROWSIZE += 1
 endw

;_______________________________________________________________________________
;Program Section:
;=========================== Reset-Entry
 if BL_LOCATION == 0
;bootloader resides at beginning of flash (addr.0):
	ORG	0

	btfss	STATUS,NOT_TO
	goto	appStart		;reset by watchdog timeout: -> app.
	movlb	1
	goto	startup

;=========================== Interrupt Service Routine
;The entry point of the applications interrupt routine has to be appStart+4 !

	ORG	4
interrupt
	bra	appStart+4		;goto application's ISR	
 else
;bootloader resides at some higher memory address:
  ifdef RESVECTOR
	ORG 	RESVECTOR		;case build for loading by bootloader
  else
	ORG	0			;for loading by programmer device
  endif

  if BL_LOCATION >= 0x800		;(must set PCLATH before goto)
	movlp	high BL_LOCATION
  endif
	goto	startup			;jump to bootloader's reset entry 

;The applications interrupt routine is at it's original address (0x004)!

;*************************************************************************************
	ORG	BL_LOCATION
	dw	BL_END+LOGROWSIZE	;used by PICLoader tool
 endif			;(BL_LOCATION == 0)

;======================= MPU Startup 
startup	
 if BL_LOCATION > 0			;(else these 3 instructions are located @addr. 0)
	btfss	STATUS,NOT_TO
	bra	appStart		;reset by watchdog timeout: start into app.
	movlb	1
 endif
					;(bank 1 selected)
	movlw	b'00100100'		;longest timeout periode/off
	movwf	WDTCON			;to avoid WDT-reset during bl-operation

	movlw	b'01111010'		;clock = 16MHz, internal osc.
	movwf	OSCCON

	movlw	1<<VALIDAPP
	movwf	flags			;default: app is valid
	btfsc	FSR0H,7			;entry from RESET?
	goto	blFromApp		;no, called from application (app must set FSR0H,7)

 ifdef ENABLE_CHECKSUM
	banksel	PMADRL
    ifdef EECON1			;if there is EEPROM memory:
	bsf	EECON1,EEPGD		;program memory access
    endif

;-------------- test for valid application
  if BL_LOCATION == 0
	movlw	low (appStart+3)	;flash word containing length of application code 
	movwf	PMADRL			
	movlw	high (appStart+3)
	movwf	PMADRH
	clrf	FSR0H			;FSR0 = &appLength
	movlw	appLength
	movwf	FSR0L
	call	FlashRead		;appLength = value from flash-1 
					;(FSR0 -> chksum, thus first FlashRead will initialize chksum)
    if (low appStart) != 0
	movlw	low appStart		;set PMADR = start of application flash
	movwf	PMADRL
    else
	clrf	PMADRL
    endif

  else				;(BL_LOCATION > 0)
	movlw	low (appStart+6)	;memory containing application length
	movwf	PMADRL
	movlw	high (appStart+6)
	movwf	PMADRH
	clrf	FSR0H
	movlw	appLength
	movwf	FSR0L
	call	FlashRead		;[appstart+6] -> appLength
	call	FlashRead		;[appstart+7] -> chksum (init), appLength--

	clrf	PMADRL			;start checksum calculation @0x000...
	clrf	PMADRH
  endif				;(BL_LOCATION == 0)

  ifdef	SIMULATOR
;	goto	blFromApp		;skip checking app. if simulator
  endif
chkAppLp
	call	FlashRead		;add one FLASH word in each loop
	bcf	FSR0L,1			;set FSR0 back -> dummy	(don't use addfsr here!)
	btfss	appLength+1,7
	goto	chkAppLp		;(appLength-- >= 0)
	
	movf	chksum+1,W
	andlw	0x3F			;14 bit checksum must give 0
	iorwf	chksum,W
	btfss	STATUS,Z		;skip "clrf flags" if app is valid
 endif				;(ENABLE_CHECKSUM)

blFromApp
	clrf	flags			;invalidate app

	banksel ANSELA	
	clrf	ANSELA		;all digital...
   ifdef ANSELB
	clrf	ANSELB
   endif
   ifdef ANSELC
	clrf	ANSELC
   endif

 ; ---------- if used: initialize hardware uart (EUSART)
 ifdef EUSART
	clrf	halfDplxCnt
	movlw	low BAUDGEN
	movwf	SPBRGL
    if (high BAUDGEN)
	movlw	high BAUDGEN
	movwf	SPBRGH
    else
	clrf	SPBRGH
    endif
	movlw	1<<BRGH	| 1<<TXEN   	;high speed baudrate, enable Tx 
	movwf	TXSTA
	movlw	1<<SPEN | 1<<CREN   	;enable serial Tx & Rx
	movwf	RCSTA
	movlw	1<<BRG16
	movwf	BAUDCON
   ifdef ODCONA
     ifdef COM_PORT			;if single wire/half duplex:
	banksel	ODCONA			
	movlw	1<<COM_PIN
	movwf	COM_PORT    		;set USART Tx Pin to Open Drain
     endif
   endif
   ifdef RXPPS
	banksel RXPPS		    	; pin is selectable via PPS:
	movlw	((RX_PORT-PORTA)<<3) | RX_PIN
	movwf	RXPPS		    	;USART Rx Pin selection
	banksel	RA0PPS
	movlw	0x14		    	;src = USART Tx
	movwf	RA0PPS+((TX_PORT-PORTA)<<3)+TX_PIN
   endif
	banksel	TRISA
   ifndef COM_PORT			;full duplex communication:
	bcf	TX_PORT,TX_PIN		;Tx pin = output
   endif 
	
 else ;(not EUSART:)
   ifndef COM_PORT			;full duplex communication:
	banksel	LATA
	bsf	TX_PORT,TX_PIN		;Tx pin = output high (idle)
	banksel	TRISA
	bcf	TX_PORT,TX_PIN	
   else
	banksel	TRISA
   endif 
 endif ;(EUSART)

  ifdef LED_PORT
     ifdef NLED_PIN
	bcf	LED_PORT,NLED_PIN	;(-> TRISx) LED => output
     else
	bcf	LED_PORT,LED_PIN	;(-> TRISx) LED => output
     endif
  endif
  ifdef EUSART
	movlw	b'11000110'		;WPU:dis, TMR0: T = 32s 
  else
	movlw	b'11000000'|T0_PSV	;WPU:dis, TMR0: PS from baudrate calc. 
  endif
	movwf	OPTION_REG

	btfss	flags,VALIDAPP
	goto	loader			;no valid app: jump into bootloader

;we have a valid app, but maybe user wants to start bootloader:
	banksel	0
 ifdef BOOT_PORT
   ifdef NBOOT_PIN
	btfsc	BOOT_PORT,NBOOT_PIN	;holding bootpin low while reset starts bootloader
   else
	btfss	BOOT_PORT,BOOT_PIN	;holding bootpin high while reset starts bootloader
   endif
 else 
	btfss	RX_PORT,RX_PIN
 endif
	bra	appStart		;serial Rx-pin is low: regular start into app

loader
	call	sendID
loaderLp
	call	blUartRx
	btfss	timeoutCnt+1,7		;no timeout if negative
	bra	appStart		;timeout: start application
	addlw	4			;valid cmd -> 0..3
	btfsc	STATUS,C		;ignore invalid commands
	call	processCmd		;process valid command (0xFC..0xFF)
	goto	loaderLp		;invalid or command complete

;=========================== process BL command byte 
;input: valid command index in W (0..3)  
processCmd
	bsf	flags,RDFLG		;default: read
	brw				;cmd:
	goto	readWord		;0xFC: read data
	goto	writeRow		;0xFD: write following block of data to flash	
	goto	sendID			;0xFE: bl-version
	reset				;0xFF: execute reset

;-------------------- Command: READ
returnDataWord				;send memory data to host:
	call	FlashRead		;read memory
	moviw	-2[FSR0]
	call	blUartTx		;(data lsb)		
	moviw	-1[FSR0]
	goto	blUartTx		;(data msb)
	
;================ BL command processing: set flash address
writeRow
	bcf	flags,RDFLG		;change to "write"
readWord
	clrf	chksum			;init packet checksum (for cmd WRITE, W=1)

	call	blUartRx		;get low address
	movwf	PMADRL			;set address register low

	clrf	FSR0L			;FSR0 -> start of data buffer 
	movlw	0x20			;(use linear data memory addressing)
	movwf	FSR0H

	call	blUartRx		;get high address
	movwf	highaddr	
	movwf	PMADRH			;set address register high

 ifdef	EECON1				;if there is EEPROM memory:
	xorlw	0xC0			;set EEPG (=bit 7) and CFGS (=bit 6)
	andlw	0xC0			;according to given MSBs of address: 
	btfsc	WREG,7			;10.. = cfg.-> 01, 11.. = EEPROM -> 00
	bcf	WREG,6			;00.., 01.. = program memory -> 10
	movwf	EECON1
 else
	clrf	PMCON1
	btfsc	highaddr,7		;access to config words?
	bsf	PMCON1,CFGS		;yes: set CFGS-bit
 endif
	btfsc	flags,RDFLG		;read command?
	goto	returnDataWord		;y

	movlw	~(ROWSIZE-1)
	andwf	PMADRL,F		;for sure: set address to start of write block

;-------------------- Cmd: WRITE, get full row of data
;Receive data bytes until a full row of data has been received		
blGetData
	call	blUartRx		
	movwi	FSR0++			;write data into buffer

	btfss	FSR0L,LOGROWSIZE+1	;got full row of data?
	goto	blGetData		;no, read more data...

;------------------ got a full row of data to be written into flash:
	clrf	FSR0L			;reset to start of row
	call	blUartRx		;add chksum: Z=1 if sum == 0		
	btfss	STATUS,Z		
	goto	sendNAK			;wrong checksum: send NAK, don't write to flash
		
	btfsc	highaddr,7		;writing config(user-ID) or EEPROM?
	goto	flashWriteBuf		;y: no risk to overwrite bootloader
					;(Z=1 if highaddress == 0)
 if BL_LOCATION == 0
;bootloader is located at start of flash.
;writing data to address < APP_START would overwrite BL code: reject this! 
	movlw	high FLASHSIZE-1	;avoid erroneous erasing of bootloader by 
	andwf	highaddr,F		;wrapping arround from address > flashsize!
   if APP_START == 0x100		
	btfsc	STATUS,Z		;highaddr == 0 would overwrite bootloader 
	goto	sendNAK			;address < 0x0100: send NAK
   else
	movlw	low APP_START		;APP_START is not 0x100: 
	subwf	PMADRL,W		;compare full address 
	movlw	high APP_START
	subwfb	highaddr,W
	btfss	STATUS,C
	goto	sendNAK			;address < APP_START: send NAK
   endif

 else					;(BL_LOCATION > 0)
;bootloader is located in upper memory:
     ifndef BL_ON_TOP		; else writing above bootloader will be rejected anyway
	movlw	high FLASHSIZE-1	;avoid erroneous erasing of bootloader by 
	andwf	highaddr,F		;wrapping arround from address > flashsize!
     endif
	movf	highaddr,W			
	iorwf	PMADRL,W		;Z=1 if address == 0 (keep bootloader's reset vector!) 
	btfss	STATUS,Z
	goto	writeAppRow		; not writing reset vector area 

;copy first 1 or 2 words from flash row (addr.=0) into buffer, then flashWriteBuf:
saveResetVector
	call	FlashRead		;read 1st word -> FSR0++
   if BL_LOCATION >= 0x800		;must save two words! (movlp, goto)
	call	FlashRead		;read 2nd word -> FSR0++
   endif
	clrf	PMADRL			;-> start of row
	clrf	FSR0L

writeAppRow
   if (low BL_LOCATION) == 0
	movlw	high BL_LOCATION	;0xVV00: compare only higher address byte
	subwf	highaddr,W
   else
	movlw	low BL_LOCATION
	subwf	PMADRL,W
	movlw	high BL_LOCATION
	subwfb	highaddr,W
   endif
   ifdef BL_ON_TOP			;if there is no reverved memory above bootloader:
	btfsc	STATUS,C		;accept writing < BL_LOCATION
	goto	sendNAK			;reject writing >= BL_LOCATION
   else
	btfss	STATUS,C		;not BL_ON_TOP: accept writing above bootloader
	goto	flashWriteBuf		;address < BL_LOCATION: write row, send ACK

	movlw	low BL_END
	subwf	PMADRL,W
	movlw	high BL_END
	subwfb	highaddr,W
	btfss	STATUS,C
	goto	sendNAK			;address < BL_END: must not overwrite bootloader!
    endif 	;BL_ON_TOP
 endif	;(BL_LOCATION == 0)


;***************************************************************************
; Routines for Read/Write Flash Memory
;***************************************************************************

;========================== Erase flash row, then write data from RAM into flash
;PMADR = destination address (row address)

flashWriteBuf
	bsf	PMCON1,FREE		;(erase)
	bsf	PMCON1,WREN 		;(enable wr)
	call	FlashWriteTrigger	;erase row...

FlashWriteLp
	moviw	FSR0++  		; Load first data byte into lower
	movwf	PMDATL  		;
	moviw 	FSR0++  		; Load second data byte into upper
	movwf 	PMDATH  		;

	bsf 	PMCON1,LWLO  		; default:  Only Load Write Latch
	btfss	FSR0L,LOGROWSIZE+1	; skip on last word -> program!
	btfsc	PMCON1,CFGS		; do not skip if writing config
	bcf 	PMCON1,LWLO  		; write config(ID) immediately, or last word
	
	call	FlashWriteTrigger
	incf	PMADRL,F  		; address++

	btfss	FSR0L,LOGROWSIZE+1	; skip on last word: operation complete
	goto	FlashWriteLp		; no: Write next latches

	bcf	PMCON1,WREN  		; Disable writes
sendACK
	movlw	ACK			;notify host: "ok+ready for next command"
	goto	blUartTx


FlashWriteTrigger
	movlw	0x55  			; Start of required write sequence:
	movwf	PMCON2  		; Write 55h
	movlw	0xAA  			;
	movwf	PMCON2  		; Write AAh
	bsf	PMCON1,WR  		; Set WR bit to begin write
	nop  				; NOP instructions are forced as processor
	nop				; loads program memory write latches
 ifdef EECON1
	btfsc	EECON1,WR		; in case of writing to EEPROM:
	goto	$-1			; wait for write op. complete
 endif
	return

;========================== Read data from flash -> FSR0++
;PMADR-reg. = source address (postinc.)
;adds data to chksum if checking is enabled
;decrements appLength
FlashRead
	bsf	PMCON1,RD
	nop
	nop
 ifdef ENABLE_CHECKSUM	
   if(BL_LOCATION > 0)
;in case of appLength gets loaded by this call, it contains value from flash
	movlw	-1
	addwf	appLength,F		;appLength--
	addwfc	appLength+1,F		
   endif
	movf	PMDATL,W
	addwf	chksum,F		;chksum += PMDAT
	movwi	FSR0++			;*(FSR0++) = PMDAT
	movf	PMDATH,W
	addwfc	chksum+1,F
	movwi	FSR0++			

   if(BL_LOCATION == 0)
;in case of appLength gets loaded by this call, it contains value from flash -1
	movlw	-1
	addwf	appLength,F		;appLength--
	addwfc	appLength+1,F		
   endif
 else		
	movf	PMDATL,W
	movwi	FSR0++			;no checksum and loop counter,
	movf	PMDATH,W		;just store data
	movwi	FSR0++
 endif			;(ENABLE_CHECKSUM)	
	incfsz	PMADRL,F		;address++
	return
	incf	PMADRH,F
	return

;***************************************************************************
;  UART Tx/Rx Routines
;***************************************************************************

;================== Receive a byte from serial port

blUartRx
 ifdef SIMULATOR
	banksel	PMADRL
	movwf	s_data		;(for debugging)
	btfsc	flags,7
	return
 endif
	banksel	0
	LEDOFF

;---------------- EUSART Receive Byte
 ifdef EUSART
	movlw	0x80			;set timeout = approx. 1 sec
	movwf	timeoutCnt+1		

blUartRx1
	btfsc	PIR1,RCIF		;got a byte in Rx buffer?
	goto	blRxGotByte		;y
	
	btfss	flags,VALIDAPP		;check for timeout condition:
	goto	blUartRx1		;no valid app: timeout is disabled
	btfss	INTCON,TMR0IF		;Timeout of TMR0?
	goto	blUartRx1		
	bcf	INTCON,TMR0IF		;y: next timeout after 8 ms
	incfsz	timeoutCnt+1,F
	goto	blUartRx1		;1 sec. not yet over 
;	return
;(timeout: fall through into receive - receives garbage, but avoids extra "return")
	
blRxGotByte
	banksel	RCREG
	movf	RCREG,W			;read byte from USART Rx data register
	decfsz	halfDplxCnt,F
	goto	blUartRx		;ignore echoed bytes!
	incf	halfDplxCnt,F		;restore 1
 ifdef LED_PORT
	banksel	0
	LEDON
 endif
	addwf	chksum,F		;and update checksum
	banksel	PMADRL
	return				;(Z=1 if checksum == 0)
		
 else
 ;---------------- Software UART Receive Byte

	clrf	TMR0
blUartRx0
	btfsc	TMR0,7
	reset				;Rx low for longer than stopbit: no COM connection!

	btfss	RX_PORT,RX_PIN	
	goto	blUartRx0		;case it's still low: didn't receive stopbit yet 

blUartRx1
	movlw	-64/T0_DIV		;set timeout = 1 sec
	movwf	timeoutCnt+1

uartRxWaitStart
	btfss	RX_PORT,RX_PIN
	goto	RxStartBit		;detected RXD = low (startit)
	btfss	flags,VALIDAPP
	goto	uartRxWaitStart		;no valid app: timeout is disabled

	btfss	INTCON,TMR0IF		;Timeout of TMR0?
	goto	uartRxWaitStart	
	bcf	INTCON,TMR0IF		
 if LOADERBAUD > 60000			
 	btfss	RX_PORT,RX_PIN
	goto	RxStartBit		;detected RXD = low (startit)
 endif	
	incfsz	timeoutCnt,F
	goto	uartRxWaitStart		;16 ms not yet over (@T0CLK = 4MHz)
	incfsz	timeoutCnt+1,F
	goto	uartRxWaitStart		;xx times 16ms not yet over
;	return
;(timeout: fall through into receive - receives garbage, but avoids extra "return")
RxStartBit
	movlw	-(BITTIME/2 -(15/T0_DIV)) ;prepare timer for scanning input level 
	movwf	TMR0			;in the middle of bit frame

	movlw	0x80
	movwf	s_data

uartRxLoop
	call	waitBitTime		;reload timer, wait for timeout

	lsrf	s_data,F
	btfsc	RX_PORT,RX_PIN
	bsf	s_data,7		;data bit -> MSB

	btfss	STATUS,C		;got all data bits? 
	goto	uartRxLoop		;not yet: repeat...

	LEDON

	movf	s_data,W
	addwf	chksum,F
	banksel	PMADRL
	return				;(Z=1 if checksum == 0)
 
;reload UART bit timer, wait for timeout
waitBitTime
	banksel	0
	movlw	BITTIME			;reload timer for next bit
	subwf	TMR0,F
	bcf	INTCON,TMR0IF
	btfss	INTCON,TMR0IF		;wait for timeout
	goto	$-1
	return
 endif		; (EUSART)

sendID
	call	sendACK
 ifdef ENABLE_CHECKSUM
	movlw	BL_VERSION | 0x80	;set MSB if application checksum is used
 else
	movlw	BL_VERSION
 endif
	goto	blUartTx

sendNAK
	movlw	NAK

;================== Transmit byte (W) to serial port 
blUartTx
 ifdef EUSART
;---------------- EUSART Transmit Byte
   ifdef COM_PORT			;single wire/half duplex:
	clrf	temp
	incfsz	temp,F			;force a little break after previous Rx op.
	goto	$-1
   endif
	
	banksel	TXREG
	btfss	TXSTA,TRMT   
	goto	$-1			;wait if last byte not yet sent
	movwf	TXREG
   ifdef COM_PORT			;single wire/half duplex:
	incf	halfDplxCnt,F
   endif
	return
 else
;---------------- Software UART Transmit Byte
	movwf	s_data
	call	waitBitTime		;force a little break after previous Rx op.,
					;banksel 0
	movlw	9
	movwf	temp			;loop counter

	bcf	TX_PORT,TX_PIN		;port = low for startbit
  ifdef COM_PORT			;in case of single wire/half duplex:
	banksel	TRISA
	bcf	TX_PORT,TX_PIN		;port = output
  endif

blUartTxLp				;send next data bit:
	call	waitBitTime	
   ifdef COM_PORT			;in case of single wire/half duplex:
	banksel	TRISA			;emulate "open drain" output
   endif
	movbit 	s_data,0,TX_PORT,TX_PIN

	bsf	STATUS,C		
	rrf	s_data,F		;MSB = 1 (-> stopbit)

	decfsz	temp,F			;all bits done?
	goto	blUartTxLp		;no, next bit...
	return
 endif		;(EUSART)

BL_END	equ (($-1) & ~(ROWSIZE-1)) + ROWSIZE

 if BL_END > CODEROMSIZE
	error "Code exceeds CODEROMSIZE!"
 endif

;*******************************************************************************
; jump into application 
;*******************************************************************************
;After bootloader was flashed by another bootloader, the installation code
;will be executed from the just intstalled bootloader 

 if BL_LOCATION == 0
appStart 	org	APP_START
	goto   	install			
	nop			
	dw	0			;application checksum data
	dw	ENDINST+1-appStart	;length of checksum range 
					;(including checksum word -> +1)		
 else
;appStart org	PATCH_ROW-8+ROWSIZE	;(was BL version 0)
appStart org	BL_LOCATION-8
	movlp	0			;PC program will patch the application's reset entry here!
	goto	install

	org	BL_LOCATION-2
	dw	ENDINST+1		;length of checksum range 
					;(including checksum word -> +1)		
	dw	0			;application checksum data
 endif

;*******************************************************************************
; Installation routine
;*******************************************************************************

; check if reset vector needs to be adjusted.
; Modify reset vector if it does not point to new bootloader's start.

;----- data used by PICLoader to prepare the installer application:
;(i.e. calculate checksum, else the bootloader wouldn't execute it) 
	ORG	0x104
;signature to identify installer code (8 words):
	db	0,"!",0,"L",0,"d",0,"r",0,"I",0,"n",0,"s",0,"!"		
;signature to identify installer code
 if BL_LOCATION == 0
	dw	appStart		;installer checksum base
	dw	ENDINST-appStart	;length of checksum range 
 else
	dw	0			;installer checksum base
	dw	ENDINST			;length of checksum range 
 endif
	dw	HARDWARE_ID		;must match controller type, otherwise
					;BL installation will be rejected

;----- data used by installer
ResetCC					;copy of reset code:
 if BL_LOCATION == 0	
resetCCLen equ 4
	btfss	STATUS,NOT_TO
	goto	appStart	
	movlb	1
	goto	startup
 else
   if BL_LOCATION >= 0x800		;(must set PCLATH before goto)
resetCCLen equ 2
	movlp	high BL_LOCATION
   else
resetCCLen equ 1
   endif
	goto	startup	
 endif

;------ installer program:
install
	banksel	0
	LEDON
	movlw	resetCCLen
	movwf	temp
	banksel	PMCON1
	clrf	PMCON1
    ifdef EECON1			;if there is EEPROM memory:
	bsf	EECON1,EEPGD		;set program memory access
    endif

;compare current reset vector with data (ResetCC):
inst_cmpLp
	clrf	PMADRH
	decf	temp,W
	movwf	PMADRL			;-> MPU reset vector
	bsf	PMCON1,RD		;read flash
	nop
	nop

	movf	PMDATL,W		
	movwf	dummy			;save current vector -> dummy
	movf	PMDATH,W		
	movwf	dummy+1

	movlw	high ResetCC		;select data to compare with
	movwf	PMADRH
	movlw	low ResetCC-1
	addwf	temp,W
	movwf	PMADRL
	bsf	PMCON1,RD		;read flash
	nop
	nop

	movf	PMDATL,W
	xorwf	dummy,F	
	movf	PMDATH,W
	xorwf	dummy+1,W
	iorwf	dummy,W
	btfss	STATUS,Z		;equal?	
	goto	instAdjustVect		;no, must adjust reset vector

	decfsz	temp,F
	goto	inst_cmpLp		;compare next words
	goto	instComplete		;reset vector is already the right one!

;----------- adjust reset vector to jump into new bootloader
instAdjustVect
	call	ReadRow0		;copy first flash row -> RAM
	movlw	high ResetCC		;select new reset vector data
	movwf	PMADRH
	movlw	low ResetCC+resetCCLen
	movwf	PMADRL
	
	movlw	resetCCLen*2
	movwf	FSR0L
AdjVectLp
	decf	PMADRL,F
	bsf	PMCON1,RD		;read flash
	nop
	nop
	movf	PMDATH,W
	movwi	--FSR0			;replace data in RAM
	movf	PMDATL,W
	movwi	--FSR0

	movf	FSR0L,F			; == 0 ?
	btfss	STATUS,Z
	goto	AdjVectLp		;copy next word

	call	WriteRow0

;------------ install complete, jump into bootloader
instComplete
	banksel	0
	LEDOFF
	bsf	FSR0H,7			;stay in bootloader
	goto	0



;---------- copy row0 into RAM
ReadRow0
	banksel	PMADRL
	clrf	FSR0L
	movlw	0x20
	movwf	FSR0H
	clrf	PMADRL
	clrf	PMADRH

copyRow0Lp
	bsf	PMCON1,RD
	nop
	nop
	movf	PMDATL,W
	movwi	FSR0++			;*(FSR0++) = PMDAT
	movf	PMDATH,W
	movwi	FSR0++			
	incf	PMADRL,F		;address++
	btfss	PMADRL,LOGROWSIZE	;got full row of data?
	goto	copyRow0Lp		;no, loop
	return

;---------- write contents of row0 to flash
WriteRow0
	banksel	PMADRL
	clrf	FSR0L
	movlw	0x20
	movwf	FSR0H
	clrf	PMADRL
	clrf	PMADRH

	bsf	PMCON1,FREE		;(erase)
	bsf	PMCON1,WREN 		;(enable wr)
	movlw	0x55  			; Start of required write sequence:
	movwf	PMCON2  		; Write 55h
	movlw	0xAA  			;
	movwf	PMCON2  		; Write AAh
	bsf	PMCON1,WR  		; Set WR bit to begin write
	nop  			
	nop				

WriteRow0Lp
	moviw	FSR0++  		; Load first data byte into lower
	movwf	PMDATL  		;
	moviw 	FSR0++  		; Load second data byte into upper
	movwf 	PMDATH  		;

	bsf 	PMCON1,LWLO  		; default:  Only Load Write Latch
	btfsc	FSR0L,LOGROWSIZE+1	; on last word -> program!
	bcf 	PMCON1,LWLO  		; last word: program
	
	bsf	PMCON1,WREN 		;(enable wr)
	movlw	0x55  			; Start of required write sequence:
	movwf	PMCON2  		; Write 55h
	movlw	0xAA  			;
	movwf	PMCON2  		; Write AAh
	bsf	PMCON1,WR  		; Set WR bit to begin write
	nop  			
	nop				

	incf	PMADRL,F  		; address++
	btfss	FSR0L,LOGROWSIZE+1	; skip on last word: operation complete
	goto	WriteRow0Lp		; Write next latches

	bcf	PMCON1,WREN  		; Disable writes
	return



ENDINST dw 	0,0			;end of installer code, used for checksum generation
					;storage for +checksum, -checksum, so over-all 
					;checksum is not changed
	END